// Copyright 2012-2013 Adobe Systems Incorporated.  All rights reserved.

//--------------------------------------------------------------------
// CLASS:
//   JQuery.DesignTime.Widget.Dialog
//
// DESCRIPTION:
//   This class is used by the Dialog property inspector to manage
//   and manipulate the design-time state of the widget's markup in the
//   design view.
//--------------------------------------------------------------------

//--------------------------------------------------------------------
// FUNCTION:
//   Dialog
//
// DESCRIPTION:
//	 This is the constructor method for this design time Dialog object.
//	 It calls the base widget constructor which helps us inherit all base methods
//	 like addClass, removeClass, ensureValidElements etc
//	 
//   It also manages the Dialog's widget markup in the design view. This constructor
//   is registered in the JQuery Widget translator found at:
//
//       Configuration/Translators/jQueryWidget.htm
//	 
//
// ARGUMENTS:
//   dom - object - The DOM used by the document in design view.
//   element - object - The top-level DOM element for our widget markup.
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog = function(dom, element, consArgs)
{
  JQuery.DesignTime.Widget.Base.call(this, dom, element);

	this.opts = {};
	this.opts = consArgs;
	this.triggerChanged = false;
	this.triggerElement = "";
	this.triggerEvent = "click";
	this.init(element);
};

// JQuery.DesignTime.Widget.Dialog derives from our JQuery.DesignTime.Widget.Base
// class, so create a Base object and use it as our prototype so we inherit all of its
// methods.
JQuery.DesignTime.Widget.Dialog.prototype = new JQuery.DesignTime.Widget.Base();

// Reset the constructor property back to Dialog for our newly created prototype
// object so callers know that our object was created with the Dialog constructor.
JQuery.DesignTime.Widget.Dialog.prototype.constructor = JQuery.DesignTime.Widget.Dialog;

//--------------------------------------------------------------------
// FUNCTION:
//   init
//
// DESCRIPTION:
//   Initializes the design-time object's properties. This  method is
//   called from the constructor and refresh() methods.
//
// ARGUMENTS:
//   element - object - The widget's top-level DOM element.
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------
JQuery.DesignTime.Widget.Dialog.prototype.init = function(element)
{
	this.element = this.getElement(element);
	this.positionArray = [];
	this.animationArray = [];
	this.elementArray = [];
	this.eventArray = [];
	this.initPositionArray();
	this.initAnimationArray();
	this.initElementArray();
	this.initEventArray();
	
	if (!this.opts)
		this.opts = {};
		
	var consArgs = this.getConstructorArgs('dialog');
	this.opts = consArgs; 
	
	this.canRefresh = true;
};

JQuery.DesignTime.Widget.Dialog.prototype.recalculateOpts = function()
{
	this.opts = {};
	var consArgs = this.getConstructorArgs(this.widgetType);
	this.opts = consArgs;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getTitleValue
//
// DESCRIPTION:
//  	This method returns the value of the title attribute in the widget constructor.
//		If user changes the title value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	String: The value of the ButtonImage attribute as string
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getTitleValue = function()
{
	if( this.opts && this.opts.title != null && typeof this.opts.title != 'undefined' )
	{
		return this.opts.title;
	}
	return "";
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setTitleValue
//
// DESCRIPTION:
//  	When a new value for title is attained for the title attribute from the PI
//		we have to update our opts object with this new title value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setTitleValue = function(titleValue)
{
	if( titleValue == null || typeof titleValue == "undefined" )
		return;
	
	if (!this.opts)
		this.opts = {};
	
	var oldTitleValue = this.opts.title;
	this.opts.title = titleValue;
	
	if( oldTitleValue != this.opts.title)
		this.updateOptions(this.widgetType, this.opts);
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getPositionIndex
//
// DESCRIPTION:
//  	This method will return the index of the position List element
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Number: The value of the index of position list
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getPositionIndex = function()
{
	if( this.opts && this.opts.position != null && typeof this.opts.position != 'undefined' )
	{
		var positionArray = this.getPositionArray();
		for( var i = 0; i < positionArray.length; i++ )
		{
			if( this.opts.position.toString() == positionArray[i] )
				return i;
		}
	}
	return 0;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setPositionIndex
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the position
//		list.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setPositionIndex = function(positionIndex)
{
	if( positionIndex == null || typeof positionIndex == "undefined" )
		return;
		
	if (!this.opts)
		this.opts = {};
	
	var oldPostionValue = this.opts.position;
	
	if( positionIndex == 0 )
		this.opts.position = null;
	else
		this.opts.position = this.getPositionArray()[positionIndex];
	
	if( oldPostionValue != this.opts.position )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getWidthValue
//
// DESCRIPTION:
//  	This method returns the value of the width attribute in the widget constructor.
//		If user changes the width value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	String: The value of the width attribute as string
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getWidthValue = function()
{
	if( this.opts && this.opts.width != null && typeof this.opts.width != 'undefined' )
	{
		return this.opts.width;
	}
	return 300;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setWidthValue
//
// DESCRIPTION:
//  	When a new value for width is attained for the width attribute from the PI
//		we have to update our opts object with this new width value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setWidthValue = function(widthValue)
{
	if( widthValue == null || typeof widthValue == "undefined" )
		return;
	
	if (!this.opts)
		this.opts = {};
	
	var oldwidthValue = this.opts.width;
	
	if( widthValue == "" || parseInt(widthValue).toString() == 'NaN' || widthValue == 300 )
	{
		this.opts.width = null;
	}	
	else
	{
		this.opts.width = parseInt(widthValue);
	}
	
	if( oldwidthValue != this.opts.width )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getHeightValue
//
// DESCRIPTION:
//  	This method returns the value of the height attribute in the widget constructor.
//		If user changes the height value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	String: The value of the height attribute as string
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getHeightValue = function()
{
	if( this.opts && this.opts.height != null && typeof this.opts.height != 'undefined' )
	{
		return this.opts.height;
	}
	return "auto";
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setHeightValue
//
// DESCRIPTION:
//  	When a new value for height is attained for the height attribute from the PI
//		we have to update our opts object with this new height value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setHeightValue = function(heightValue)
{
	if( heightValue == null || typeof heightValue == "undefined" )
		return;
	
	if (!this.opts)
		this.opts = {};
	
	var oldheightValue = this.opts.height;
	
	if( heightValue == "" || parseInt(heightValue).toString() == 'NaN' || heightValue == 'auto' )
	{
		this.opts.height = null;
	}	
	else
	{
		this.opts.height = parseInt(heightValue);
	}
	
	if( oldheightValue != this.opts.height )
		this.updateOptions(this.widgetType, this.opts);	
}


//--------------------------------------------------------------------
// FUNCTION:
//  	getMaxWidthValue
//
// DESCRIPTION:
//  	This method returns the value of the maxWidth attribute in the widget constructor.
//		If user changes the maxWidth value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	String: The value of the maxWidth attribute as string
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getMaxWidthValue = function()
{
	if( this.opts && this.opts.maxWidth != null && typeof this.opts.maxWidth != 'undefined' )
	{
		return this.opts.maxWidth;
	}
	return "";
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setMaxWidthValue
//
// DESCRIPTION:
//  	When a new value for maxWidth is attained for the maxWidth attribute from the PI
//		we have to update our opts object with this new maxWidth value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setMaxWidthValue = function(maxWidthValue)
{
	if( maxWidthValue == null || typeof maxWidthValue == "undefined" )
		return;
	
	if (!this.opts)
		this.opts = {};
	
	var oldmaxWidthValue = this.opts.maxWidth;
	
	if( maxWidthValue == "" || parseInt(maxWidthValue).toString() == 'NaN' )
	{
		this.opts.maxWidth = null;
	}	
	else
	{
		this.opts.maxWidth = parseInt(maxWidthValue);
	}
	
	if( oldmaxWidthValue != this.opts.maxWidth )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getMaxHeightValue
//
// DESCRIPTION:
//  	This method returns the value of the maxHeight attribute in the widget constructor.
//		If user changes the maxHeight value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	String: The value of the maxHeight attribute as string
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getMaxHeightValue = function()
{
	if( this.opts && this.opts.maxHeight != null && typeof this.opts.maxHeight != 'undefined' )
	{
		return this.opts.maxHeight;
	}
	return "";
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setMaxHeightValue
//
// DESCRIPTION:
//  	When a new value for maxHeight is attained for the maxHeight attribute from the PI
//		we have to update our opts object with this new maxHeight value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setMaxHeightValue = function(maxHeightValue)
{
	if( maxHeightValue == null || typeof maxHeightValue == "undefined" )
		return;
	
	if (!this.opts)
		this.opts = {};
	
	var oldmaxHeightValue = this.opts.maxHeight;
	
	if( maxHeightValue == "" || parseInt(maxHeightValue).toString() == 'NaN' )
	{
		this.opts.maxHeight = null;
	}	
	else
	{
		this.opts.maxHeight = parseInt(maxHeightValue);
	}
	
	if( oldmaxHeightValue != this.opts.maxHeight )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getMinWidthValue
//
// DESCRIPTION:
//  	This method returns the value of the minWidth attribute in the widget constructor.
//		If user changes the minWidth value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	String: The value of the minWidth attribute as string
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getMinWidthValue = function()
{
	if( this.opts && this.opts.minWidth != null && typeof this.opts.minWidth != 'undefined' )
	{
		return this.opts.minWidth;
	}
	return 150;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setMinWidthValue
//
// DESCRIPTION:
//  	When a new value for minWidth is attained for the minWidth attribute from the PI
//		we have to update our opts object with this new minWidth value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setMinWidthValue = function(minWidthValue)
{
	if( minWidthValue == null || typeof minWidthValue == "undefined" )
		return;
	
	if (!this.opts)
		this.opts = {};
	
	var oldminWidthValue = this.opts.minWidth;
	
	if( minWidthValue == "" || parseInt(minWidthValue).toString() == 'NaN' || minWidthValue == '150' )
	{
		this.opts.minWidth = null;
	}	
	else
	{
		this.opts.minWidth = parseInt(minWidthValue);
	}
	
	if( oldminWidthValue != this.opts.minWidth )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getMinHeightValue
//
// DESCRIPTION:
//  	This method returns the value of the minHeight attribute in the widget constructor.
//		If user changes the minHeight value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	String: The value of the minHeight attribute as string
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getMinHeightValue = function()
{
	if( this.opts && this.opts.minHeight != null && typeof this.opts.minHeight != 'undefined' )
	{
		return this.opts.minHeight;
	}
	return 150;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setMinHeightValue
//
// DESCRIPTION:
//  	When a new value for minHeight is attained for the minHeight attribute from the PI
//		we have to update our opts object with this new minHeight value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setMinHeightValue = function(minHeightValue)
{
	if( minHeightValue == null || typeof minHeightValue == "undefined" )
		return;
	
	if (!this.opts)
		this.opts = {};
	
	var oldminHeightValue = this.opts.minHeight;
	
	if( minHeightValue == "" || parseInt(minHeightValue).toString() == 'NaN' || minHeightValue == '150' )
	{
		this.opts.minHeight = null;
	}	
	else
	{
		this.opts.minHeight = parseInt(minHeightValue);
	}
	
	if( oldminHeightValue != this.opts.minHeight )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getDraggableValue
//
// DESCRIPTION:
//  	This method returns the value of the draggable attribute in the widget constructor.
//		If user changes the draggable value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Boolean: The value of the draggable attribute as boolean
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getDraggableValue = function()
{	
	if( this.opts && this.opts.draggable != null && typeof this.opts.draggable != 'undefined' )
	{
		return (this.opts.draggable == true);
	}
	return true;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setDraggableValue
//
// DESCRIPTION:
//  	When a new value for draggable is attained for the draggable attribute from the PI
//		we have to update our opts object with this new draggable value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setDraggableValue = function(draggableValue)
{	
	if( draggableValue == null || typeof draggableValue == "undefined" )
		return;

	if (!this.opts)
		this.opts = {};
	
	var oldDraggableValue = this.opts.draggable;
	
	if( draggableValue == false )
	{
		this.opts.draggable = false;
	}	
	else
	{
		this.opts.draggable = null;
	}
	
	if( oldDraggableValue != this.opts.draggable )
		this.updateOptions(this.widgetType, this.opts);
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getCloseValue
//
// DESCRIPTION:
//  	This method returns the value of the closeOnEscape attribute in the widget constructor.
//		If user changes the closeOnEscape value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Boolean: The value of the closeOnEscape attribute as boolean
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getCloseValue = function()
{	
	if( this.opts && this.opts.closeOnEscape != null && typeof this.opts.closeOnEscape != 'undefined' )
	{
		return (this.opts.closeOnEscape == true);
	}
	return true;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setCloseValue
//
// DESCRIPTION:
//  	When a new value for closeOnEscape is attained for the closeOnEscape attribute from the PI
//		we have to update our opts object with this new closeOnEscape value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setCloseValue = function(closeOnEscapeValue)
{	
	if( closeOnEscapeValue == null || typeof closeOnEscapeValue == "undefined" )
		return;

	if (!this.opts)
		this.opts = {};
	
	var oldCloseValue = this.opts.closeOnEscape;
	
	if( closeOnEscapeValue == false )
	{
		this.opts.closeOnEscape = false;
	}	
	else
	{
		this.opts.closeOnEscape = null;
	}
	
	if( oldCloseValue != this.opts.closeOnEscape )
		this.updateOptions(this.widgetType, this.opts);
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getAutoValue
//
// DESCRIPTION:
//  	This method returns the value of the autoOpen attribute in the widget constructor.
//		If user changes the autoOpen value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Boolean: The value of the autoOpen attribute as boolean
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getAutoValue = function()
{	
	if( this.opts && this.opts.autoOpen != null && typeof this.opts.autoOpen != 'undefined' )
	{
		return (this.opts.autoOpen == true);
	}
	return true;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setAutoValue
//
// DESCRIPTION:
//  	When a new value for autoOpen is attained for the autoOpen attribute from the PI
//		we have to update our opts object with this new autoOpen value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setAutoValue = function(autoOpenValue)
{	
	if( autoOpenValue == null || typeof autoOpenValue == "undefined" )
		return;

	if (!this.opts)
		this.opts = {};
	
	var oldAutoValue = this.opts.autoOpen;
	
	if( autoOpenValue == false )
	{
		this.opts.autoOpen = false;
	}	
	else
	{
		this.opts.autoOpen = null;
	}
	
	if( oldAutoValue != this.opts.autoOpen )
		this.updateOptions(this.widgetType, this.opts);
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getModalValue
//
// DESCRIPTION:
//  	This method returns the value of the modal attribute in the widget constructor.
//		If user changes the modal value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Boolean: The value of the modal attribute as boolean
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getModalValue = function()
{	
	if( this.opts && this.opts.modal != null && typeof this.opts.modal != 'undefined' )
	{
		return (this.opts.modal == true);
	}
	return false;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setModalValue
//
// DESCRIPTION:
//  	When a new value for modal is attained for the modal attribute from the PI
//		we have to update our opts object with this new modal value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setModalValue = function(modalValue)
{	
	if( modalValue == null || typeof modalValue == "undefined" )
		return;

	if (!this.opts)
		this.opts = {};
	
	var oldModalValue = this.opts.modal;
	
	if( modalValue == true )
	{
		this.opts.modal = true;
	}	
	else
	{
		this.opts.modal = null;
	}
	
	if( oldModalValue != this.opts.modal )
		this.updateOptions(this.widgetType, this.opts);
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getResizableValue
//
// DESCRIPTION:
//  	This method returns the value of the resizable attribute in the widget constructor.
//		If user changes the resizable value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Boolean: The value of the resizable attribute as boolean
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getResizableValue = function()
{	
	if( this.opts && this.opts.resizable != null && typeof this.opts.resizable != 'undefined' )
	{
		return (this.opts.resizable == true);
	}
	return true;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setResizableValue
//
// DESCRIPTION:
//  	When a new value for resizable is attained for the resizable attribute from the PI
//		we have to update our opts object with this new resizable value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setResizableValue = function(resizableValue)
{	
	if( resizableValue == null || typeof resizableValue == "undefined" )
		return;

	if (!this.opts)
		this.opts = {};
	
	var oldResizableValue = this.opts.resizable;
	
	if( resizableValue == false )
	{
		this.opts.resizable = false;
	}	
	else
	{
		this.opts.resizable = null;
	}
	
	if( oldResizableValue != this.opts.resizable )
		this.updateOptions(this.widgetType, this.opts);
}

//--------------------------------------------------------------------
// FUNCTION:
//   getAssets
//
// DESCRIPTION:
//   Static function that returns the assets to be applied to page
//
// ARGUMENTS:
//   None
//
// RETURNS:
//   (array of objects)
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.getAssets = function()
{
	var assets = new Array();
	
	assetObject = new Object();
	assetObject.fullPath = jQDialogImagesPath;
	assetObject.file = jQDialogImagesFile;
	assetObject.type = "";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQCoreCssPath;
	assetObject.file = jQCoreCssFile;
	assetObject.type = "link";
	assets.push(assetObject);

	assetObject = new Object();
	assetObject.fullPath = jQCssThemePath;
	assetObject.file = jQCssThemeFile;
	assetObject.type = "link";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQDialogCssWidgetPath;
	assetObject.file = jQDialogCssWidgetFile;
	assetObject.type = "link";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQResizableCssPath;
	assetObject.file = jQResizableCssFile;
	assetObject.type = "link";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQMainPath;
	assetObject.file = jQMainFile;
	assetObject.type = "javascript";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQDialogJsPath;
	assetObject.file = jQDialogJsFile;
	assetObject.type = "javascript";
	assets.push(assetObject);
	
	return (assets);
	
};

//--------------------------------------------------------------------
// FUNCTION:
//   refresh
//
// DESCRIPTION:
//   None
//
// ARGUMENTS:
//   Syncs up the internal state of the design-time widget object
//   with the markup that currently exists in the design view.
//
//   This method gets called from the translator if a design-time
//   object already exists for a specific widget ID, and from various
//   methods in the Dialog property inspector.
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.refresh = function()
{
  if (!this.canRefresh)
	return;
	
	if (!this.opts)
		this.opts = {};
	var consArgs = this.getConstructorArgs(this.widgetType);
	
	this.opts = consArgs; 
  this.init(this.element_id);  
};

//--------------------------------------------------------------------
// FUNCTION:
//   JQuery.DesignTime.Widget.Dialog.getNewDialogSnippet
//
// DESCRIPTION:
//   Static utility function that returns a string containing the
//   markup for a complete Dialog.
//
// ARGUMENTS:
//  id - string - The string to use as the widget's id attribute.
//
// RETURNS:
//   String that is the HTML markup fragment for the panel.
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.getNewDialogSnippet = function(id)
{
  var dpSnippet = '<div id="' + id + '">Content for New Dialog Goes Here</div>';
  return dpSnippet;
};

//--------------------------------------------------------------------
// FUNCTION:
//   JQuery.DesignTime.Widget.Dialog.getNewDialogConstructorSnippet
//
// DESCRIPTION:
//   Static utility function that returns a string that contains the
//   constructor snippet for creating a JQuery widget with the specified id.
//
// ARGUMENTS:
//  id - string - The id of the widget markup to insert into the
//                constructor snippet.
//
// RETURNS:
//   String containing JS that is the constructor call to create the
//   widget.
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.getNewDialogConstructorSnippet = function(id)
{
  return ' $(function() {\n\t$( "#' + id + '" ).dialog(); \n});';
 /* $(function() {
		$( "#Dialog1" ).Dialog();
	});*/
};

//--------------------------------------------------------------------
// FUNCTION:
//  	initPositionArray
//
// DESCRIPTION:
//  	Initializes the position array with predefined values.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.initPositionArray = function()
{
	this.positionArray.push("center");
	this.positionArray.push("left");
	this.positionArray.push("right");
	this.positionArray.push("top");
	this.positionArray.push("bottom");
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getPositionArray
//
// DESCRIPTION:
//  	Return the position Array containing all possible values for 'position'
//		attribute of the constructor object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
//--------------------------------------------------------------------


JQuery.DesignTime.Widget.Dialog.prototype.getPositionArray = function()
{
	return this.positionArray;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	initAnimationArray
//
// DESCRIPTION:
//  	Initializes the position array with predefined values.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.initAnimationArray = function()
{
	this.animationArray.push("none");
	this.animationArray.push("blind");
	this.animationArray.push("bounce");
	this.animationArray.push("clip");
	this.animationArray.push("drop");
	this.animationArray.push("fade");
	this.animationArray.push("fold");
	this.animationArray.push("highlight");
	this.animationArray.push("puff");
	this.animationArray.push("pulsate");
	this.animationArray.push("scale");
	this.animationArray.push("shake");
	this.animationArray.push("slide");
	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getAnimationArray
//
// DESCRIPTION:
//  	Return the position Array containing all possible values for 'position'
//		attribute of the constructor object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
//--------------------------------------------------------------------


JQuery.DesignTime.Widget.Dialog.prototype.getAnimationArray = function()
{
	return this.animationArray;
}

JQuery.DesignTime.Widget.Dialog.prototype.getHideEffectIndex = function()
{
	if( this.opts && this.opts.hide != null && typeof this.opts.hide != 'undefined' )
	{
		if( this.opts.hide.effect && typeof this.opts.hide.effect != 'undefined')
		{
			for( var i = 0; i < this.animationArray.length; i++ )
			{
				if( this.opts.hide.effect.toString() == this.animationArray[i] )
					return i;
			}
		}
	}
	return 0;
}
JQuery.DesignTime.Widget.Dialog.prototype.setHideEffectIndex = function(listIndex)
{
	if( listIndex == null || typeof listIndex == "undefined" )
		return;
			
	if (!this.opts)
		this.opts = {};
		
	if (!this.opts.hide)
		this.opts.hide = {};
		
	var oldEffectValue = this.opts.hide.effect;	
	
	if( listIndex == 0 )
		this.opts.hide.effect = null;
	else
		this.opts.hide.effect = this.animationArray[listIndex];	
		
	if( this.opts.hide && !this.opts.hide.effect )	
	{
		this.opts.hide = null;
		this.updateOptions(this.widgetType, this.opts);	
	}
	else if( oldEffectValue != this.opts.hide.effect )	
		this.updateOptions(this.widgetType, this.opts);	
}
JQuery.DesignTime.Widget.Dialog.prototype.getHideDurationValue = function()
{
	if( this.opts && this.opts.hide != null && typeof this.opts.hide != 'undefined' )
	{
		if( this.opts.hide.duration && typeof this.opts.hide.duration != 'undefined')
		{
			return this.opts.hide.duration;
		}
	}
	return "400";
}

JQuery.DesignTime.Widget.Dialog.prototype.setHideDurationValue = function(hideValue)
{
	if (hideValue == null || typeof hideValue == "undefined")
		return;
			
	if (!this.opts)
		this.opts = {};
		
	if (!this.opts.hide)
		this.opts.hide = {};
		
	var oldDurationValue = this.opts.hide.duration;	
	
	if( hideValue == "400" || parseInt(hideValue).toString() == 'NaN')
		this.opts.hide.duration = null;
	else
		this.opts.hide.duration = parseInt(hideValue);	
		
	if( this.opts.hide && !this.opts.hide.effect && !this.opts.hide.duration )	
	{
		this.opts.hide = null;
		this.updateOptions(this.widgetType, this.opts);	
	}
	else if( oldDurationValue != this.opts.hide.duration )	
		this.updateOptions(this.widgetType, this.opts);	
}

JQuery.DesignTime.Widget.Dialog.prototype.getShowEffectIndex = function()
{
	if( this.opts && this.opts.show != null && typeof this.opts.show != 'undefined' )
	{
		if( this.opts.show.effect && typeof this.opts.show.effect != 'undefined')
		{
			for( var i = 0; i < this.animationArray.length; i++ )
			{
				if( this.opts.show.effect.toString() == this.animationArray[i] )
					return i;
			}
		}
	}
	return 0;
}
JQuery.DesignTime.Widget.Dialog.prototype.setShowEffectIndex = function(listIndex)
{
	if( listIndex == null || typeof listIndex == "undefined" )
		return;
			
	if (!this.opts)
		this.opts = {};
		
	if (!this.opts.show)
		this.opts.show = {};
		
	var oldEffectValue = this.opts.show.effect;	
	
	if( listIndex == 0 )
		this.opts.show.effect = null;
	else
		this.opts.show.effect = this.animationArray[listIndex];	
		
	if( this.opts.show && !this.opts.show.effect )	
	{
		this.opts.show = null;
		this.updateOptions(this.widgetType, this.opts);	
	}
	else if( oldEffectValue != this.opts.show.effect )	
		this.updateOptions(this.widgetType, this.opts);	
}

JQuery.DesignTime.Widget.Dialog.prototype.getShowDurationValue = function()
{
	if( this.opts && this.opts.show != null && typeof this.opts.show != 'undefined' )
	{
		if( this.opts.show.duration && typeof this.opts.show.duration != 'undefined')
		{
			return this.opts.show.duration;
		}
	}
	return "400";
}
JQuery.DesignTime.Widget.Dialog.prototype.setShowDurationValue = function(showValue)
{
	if (showValue == null || typeof showValue == "undefined")
		return;
			
	if (!this.opts)
		this.opts = {};
		
	if (!this.opts.show)
		this.opts.show = {};
		
	var oldDurationValue = this.opts.show.duration;	
	
	if( showValue == "400" || parseInt(showValue).toString() == 'NaN')
		this.opts.show.duration = null;
	else
		this.opts.show.duration = parseInt(showValue);	
		
	if( this.opts.show && !this.opts.show.effect && !this.opts.show.duration )	
	{
		this.opts.show = null;
		this.updateOptions(this.widgetType, this.opts);	
	}
	else if( oldDurationValue != this.opts.show.duration )	
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  getConstructorArgs
//
// DESCRIPTION:
//  Parses the jQuery widget constructor and creates an object of all the arguments it contains
//	This object can be used to display these value in the PI
//
// ARGUMENTS:
//   String: Widget Type of widget
//
// RETURNS:
//   An object containing all constructor arguments
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getConstructorArgs = function(widgetType)
{
	//We must create a new regular expression in this case because in widgetBase we don't match any new string through parantheses. But in this case we need a match within the matched string
	var matchRegExpString = this.element_id + "[\"']\\s*\\)\\.(?:" + widgetType + ")\\((?:\\s|.)*?(?:\\)?\\s*;)([\\S\\s]*?)(\\}\\s*\\)\\s*;\\s*)?\\}\\s*\\)\\s*;"
	var matchRegExp = new RegExp(matchRegExpString, "g");
	var scriptTags = this.dom.getElementsByTagName("script");
	for( var i = 0; i < scriptTags.length; i++ )
	{
		var sTag = scriptTags[i];
		var src = sTag.innerHTML;
		if(!src)
			continue;
			
		var matchStrings = matchRegExp.exec(src);
		if( matchStrings && matchStrings.length )
		{
			var args = JQuery.DesignTime.Editing.Utils.parseForJsOptions(matchStrings[0], widgetType);
			
			if (matchStrings.length > 1)
			{
				var triggerString = matchStrings[1];
				var triggerRegExp = new RegExp("\\$\\(\\s*[\"'](.*?)[\"']\\s*\\)\\s*\\.(.*?)\\(");
				var triggerVals = triggerRegExp.exec(triggerString);
				
				if (!triggerVals || triggerVals.length < 3 )
				{
					this.triggerElement = "";
					this.triggerEvent = "click";
				}	
				else if (triggerVals && triggerVals[1].length > 0)
				{
					this.triggerElement = triggerVals[1].slice(1, triggerVals[1].length);
					this.triggerEvent = triggerVals[2];
				}
			}
			return args;
		}
	}
	return null;
};

//--------------------------------------------------------------------
// FUNCTION:
//  updateOptions
//
// DESCRIPTION:
//  Updates the widget constructor in code to reflect the new values in the opts object
//	We go through code to locate the constructor string and replace this with a new string
//	created from the updated opts object
//
// ARGUMENTS:
//  String: Widget Type of widget
//	Object: The options object containing all constructor arguments
//
// RETURNS:
//  Nothing
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.updateOptions = function(widgetType, opts)
{
	var consRegExp = this.getConstructorRegExp();
	var scriptTags = this.dom.getElementsByTagName("script");
	for( var i = 0; i < scriptTags.length; i++ )
	{
		var sTag = scriptTags[i];
		var src = sTag.innerHTML;
		if(!src)
			continue;

		var htmlSplitStrings = src.split(consRegExp);
		if( htmlSplitStrings.length == 2 )
		{
			
			var preHtml = htmlSplitStrings[0];
			var postHtml = htmlSplitStrings[1];
			var allMatches = consRegExp.exec(src);
			
			if (!allMatches || allMatches.length == 0)
				return;
			
			var constructorString = allMatches[0];
			
			if(  constructorString && constructorString.length )
			{
				var newObjectString 			= ""
				var newConstructorString 	= "";
				var preConstructorString 	= "";
				var postConstructorString 	= "";
				newObjectString = JQuery.DesignTime.Widget.Base.replaceWithUpdatedOption(opts);
				if( newObjectString == null )
					return;
				
				constructorParseArray = JQuery.DesignTime.Editing.Utils.getContainedString (constructorString, '(', ')', true);
				
				if(!constructorParseArray)
				{
					return;
				}
				
				preConstructorString = constructorString.slice(0,constructorParseArray[1]);
				postConstructorString = constructorString.slice(constructorParseArray[2], constructorString.length);
				
				if (this.triggerChanged)
				{
					var newTriggerString = "";
					if (this.triggerElement)
					{
						if (!this.triggerEvent)
							this.triggerEvent = 'click';
						var newTriggerString = ";\n\t$( \"#" + this.triggerElement + "\" )." + this.triggerEvent + "(function(){\n \t\t$( \"#" + this.element_id + "\" ).dialog(\"open\");\n\t})"
					}
					postConstructorString = newTriggerString + ";\n});";
				}
				
				if( newObjectString == "" )
						newConstructorString = preConstructorString + "()" + postConstructorString
				else
						newConstructorString = preConstructorString + "({\n" + newObjectString + "\t})" + postConstructorString
				
				this.canRefresh = false;
				sTag.innerHTML = preHtml + newConstructorString + postHtml;
				this.canRefresh = true;		
			}
			else
			{
				if (dw.isDebugBuild())
				{
					alert("Regular expression fail!");
				}
			}

			return;
		}
	}
	
	return;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	initElementArray
//
// DESCRIPTION:
//  	Initializes the Element array with ids of all buttons in the document.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.initElementArray = function()
{
	this.elementArray = [];
	this.elementArray.push('none');
	
	if (!this.dom)
		return;
	
	var allButtons = this.dom.getElementsByTagName('button');
	var allInputs = this.dom.getElementsByTagName('input');
	
	for (var i = 0; i < allInputs.length; i++)
	{
		if (allInputs[i] && allInputs[i].getAttribute && (allInputs[i].getAttribute("type") == "button" || allInputs[i].getAttribute("type") == "submit"))
		{
			allButtons.push(allInputs[i]);
		}
	}
	
	if (!allButtons)
		return;
		
	for (var i = 0; i < allButtons.length; i++)
	{
		var currentButton = allButtons[i];
		if (currentButton && currentButton.getAttribute)
		{
			var currentId = currentButton.getAttribute('id');
			if (currentId && currentId != "")
				this.elementArray.push(currentId);
		}
	}
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getElementArray
//
// DESCRIPTION:
//  	Return the Elements Array containing all possible values for 'id'
//		of element which can trigger the dialog to open
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
//--------------------------------------------------------------------


JQuery.DesignTime.Widget.Dialog.prototype.getElementArray = function()
{
	return this.elementArray;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	initEventArray
//
// DESCRIPTION:
//  	Initializes the Event array with static values of possible events in jQuery.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.initEventArray = function()
{
	this.eventArray.push("click");
	this.eventArray.push("dblclick");
	this.eventArray.push("mouseover");
	this.eventArray.push("keydown");
	this.eventArray.push("keypress");
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getEventArray
//
// DESCRIPTION:
//  	Return the Events Array containing all possible values for 'event'
//		on which the dialog should open
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
//--------------------------------------------------------------------


JQuery.DesignTime.Widget.Dialog.prototype.getEventArray = function()
{
	return this.eventArray;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getElementIndex
//
// DESCRIPTION:
//  	This method will return the index of the element List element
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Number: The value of the index of element list
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getElementIndex = function()
{
	if (this.triggerElement && this.triggerElement != "")
	{
		var elementArray = this.getElementArray();
		for( var i = 0; i < elementArray.length; i++ )
		{
			if( this.triggerElement.toString() == elementArray[i] )
				return i;
		}
	}
	return 0;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setElementIndex
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the element
//		list.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setElementIndex = function(elementIndex)
{
	if( elementIndex == null || typeof elementIndex == "undefined" )
		return;
		
	var oldElementValue = this.triggerElement;
	var elementArray = this.getElementArray();
	
	if( elementIndex == 0 )
		this.triggerElement = "";
	else
		this.triggerElement = elementArray[elementIndex];
	
	if( oldElementValue != this.triggerElement )
	{
		this.triggerChanged = true;
		this.updateOptions(this.widgetType, this.opts);	
	}
	this.triggerChanged = false;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getEventIndex
//
// DESCRIPTION:
//  	This method will return the index of the event List event
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Number: The value of the index of event list
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.getEventIndex = function()
{
	if (this.triggerEvent && this.triggerEvent != "")
	{
		var eventArray = this.getEventArray();
		for( var i = 0; i < eventArray.length; i++ )
		{
			if( this.triggerEvent.toString() == eventArray[i] )
				return i;
		}
	}
	return 0;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setEventIndex
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the event
//		list.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Dialog.prototype.setEventIndex = function(eventIndex)
{
	if( eventIndex == null || typeof eventIndex == "undefined" )
		return;
		
	var oldEventValue = this.triggerEvent;
	var eventArray = this.getEventArray();
	
	if( eventIndex == 0 )
		this.triggerEvent = "click";
	else
		this.triggerEvent = eventArray[eventIndex];
			
	if( oldEventValue != this.triggerEvent )
	{
		this.triggerChanged = true;
		this.updateOptions(this.widgetType, this.opts);	
	}
	this.triggerChanged = false;
}

