// Copyright 2012-2013 Adobe Systems Incorporated.  All rights reserved.

//--------------------------------------------------------------------
// CLASS:
//   JQuery.DesignTime.Widget.Accordion
//
// DESCRIPTION:
//   This class is used by the Accordion property inspector to manage
//   and manipulate the design-time state of the widget's markup in the
//   design view.
//--------------------------------------------------------------------

//--------------------------------------------------------------------
// FUNCTION:
//   Accordion
//
// DESCRIPTION:
//	 This is the constructor method for this design time accordion object.
//	 It calls the base widget constructor which helps us inherit all base methods
//	 like addClass, removeClass, ensureValidElements etc
//	 
//   It also manages the Accordion's widget markup in the design view. This constructor
//   is registered in the JQuery Widget translator found at:
//
//       Configuration/Translators/jQueryWidget.htm
//	 
//
// ARGUMENTS:
//   dom - object - The DOM used by the document in design view.
//   element - object - The top-level DOM element for our widget markup.
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion = function(dom, element, consArgs)
{
  JQuery.DesignTime.Widget.Base.call(this, dom, element);

	this.opts = {};
	this.opts = consArgs;
  this.init(element);
  this.ensureValidDefaultPanel(); 
  this.attachBehaviors(this.defaultPanelId); 
};

// JQuery.DesignTime.Widget.Accordion derives from our JQuery.DesignTime.Widget.Base
// class, so create a Base object and use it as our prototype so we inherit all of its
// methods.
JQuery.DesignTime.Widget.Accordion.prototype = new JQuery.DesignTime.Widget.Base();

// Reset the constructor property back to Accordion for our newly created prototype
// object so callers know that our object was created with the Accordion constructor.
JQuery.DesignTime.Widget.Accordion.prototype.constructor = JQuery.DesignTime.Widget.Accordion;

//--------------------------------------------------------------------
// FUNCTION:
//   init
//
// DESCRIPTION:
//   Initializes the design-time object's properties. This  method is
//   called from the constructor and refresh() methods.
//
// ARGUMENTS:
//   element - object - The widget's top-level DOM element.
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.init = function(element)
{
  this.element = this.getElement(element);
  this.defaultPanelId = this.getActivePanel();
	this.eventArray = [];
	this.initEventArray();
	this.heightStyleArray = [];
	this.initHeightStyleArray();
	this.easingArray = [];
	this.initEasingArray();
	this.headerArray = [];
	this.initHeaderArray();
	this.activeHeaderArray = [];
	this.initActiveHeaderArray();
  
  //These are CSS classes applied to JQuery Accordions dynamically. 
  //Since we can't add these classes to JQuery generated code by itself (without modifying default code),
  //we are adding these classes to the design time objects only.
  this.openClass = "ui-helper-reset ui-state-default ui-state-active";
  this.closedClass = "ui-helper-reset ui-state-default";
  
  this.currentContent = null;
  this.canRefresh = true;
};

//--------------------------------------------------------------------
// FUNCTION:
//   getAssets
//
// DESCRIPTION:
//   Static function that returns the assets to be applied to page
//
// ARGUMENTS:
//   None
//
// RETURNS:
//   (array of objects)
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.getAssets = function()
{
	var assets = new Array();
	
	assetObject = new Object();
	assetObject.fullPath = jQAccordionImagesPath;
	assetObject.file = jQAccordionImagesFile;
	assetObject.type = "";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQCoreCssPath;
	assetObject.file = jQCoreCssFile;
	assetObject.type = "link";
	assets.push(assetObject);

	assetObject = new Object();
	assetObject.fullPath = jQCssThemePath;
	assetObject.file = jQCssThemeFile;
	assetObject.type = "link";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQAccordionCssWidgetPath;
	assetObject.file = jQAccordionCssWidgetFile;
	assetObject.type = "link";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQMainPath;
	assetObject.file = jQMainFile;
	assetObject.type = "javascript";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQAccordionJsPath;
	assetObject.file = jQAccordionJsFile;
	assetObject.type = "javascript";
	assets.push(assetObject);
	
	return (assets);
	
};

//--------------------------------------------------------------------
// FUNCTION:
//   refresh
//
// DESCRIPTION:
//   None
//
// ARGUMENTS:
//   Syncs up the internal state of the design-time widget object
//   with the markup that currently exists in the design view.
//
//   This method gets called from the translator if a design-time
//   object already exists for a specific widget ID, and from various
//   methods in the Accordion property inspector.
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.refresh = function()
{
  if (!this.canRefresh)
	return;
	
	this.opts = {};
	var consArgs = this.getConstructorArgs(this.widgetType);
	this.opts = consArgs;
	
  var curPanelIndex = this.getContentIndex(this.currentContent);
  if (curPanelIndex < 0)
  {
    this.ensureValidDefaultPanel();
    curPanelIndex = this.defaultPanelId;
  }
  
  this.init(this.element_id);
  this.attachBehaviors(curPanelIndex);
  
};

JQuery.DesignTime.Widget.Accordion.prototype.recalculateOpts = function()
{
	this.opts = {};
	var consArgs = this.getConstructorArgs(this.widgetType);
	this.opts = consArgs;
}

//--------------------------------------------------------------------
// FUNCTION:
//   getContentIndex
//
// DESCRIPTION:
//   Returns the index of the content element within the Accordion widget.
//
// ARGUMENTS:
//   contentElement - object - The DOM element that represents the content div tag
//                    within the widget's markup structure.
//
// RETURNS:
//   The index of the element within the widget or -1 if it
//   does not exist within the widget.
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.getContentIndex = function(contentElement)
{
  if (contentElement && contentElement.ownerDocument)
  {
    var contentElements = this.getContentElements();
    if (contentElements)
    {
      for (var i = 0; i < contentElements.length; i++)
      {
        if (contentElements[i] == contentElement)
          return i;
      }
    }
  }

  return -1;
};

//--------------------------------------------------------------------
// FUNCTION:
//   openPanel
//
// DESCRIPTION:
//   Opens the specified panel and closes any other panel that is
//   open. This means that we close the content div (by making display ='none')
//	 and open the new content div by setting diplay = block
//
// ARGUMENTS:
//   panel - object - The DOM element that represents the content element div that will contain the panel content
//                    within the widget's markup structure.
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.openPanel = function(panel)
{
  var contentA = this.currentContent;
  var contentB = panel;

  if (!contentA || ! contentB || contentA == contentB)
    return;

  var labelElementA = this.getLabelForContent(contentA);
  var labelElementB = this.getLabelForContent(contentB);
	
  JQuery.DesignTime.Widget.Accordion.setDisplay(contentA, "none");
  JQuery.DesignTime.Widget.Accordion.setDisplay(contentB, "block");

  this.removeClassName(labelElementA, this.openClass);
  this.addClassName(labelElementA, this.closedClass);

  this.removeClassName(labelElementB, this.closedClass);
  this.addClassName(labelElementB, this.openClass);
 
  // Add the dw show button to the panel that's closing.
    this.addShowPanelContextButton(labelElementA);
  
  // Remove the show button from the panel that's now open.
  this.removeContextButton(labelElementB);
  
  this.currentContent = contentB;
};

//--------------------------------------------------------------------
// FUNCTION:
//   onRemoved
//
// DESCRIPTION:
//   This method will be executed (or tried for execution) for any deleted widget
//	 The call will come from the translator which is listening for widget delete events.
//	 It should not lead to any error if this method is not implemented for some widget.
//	 It is a way for the widget to clean itself up before deletion.
//	 When triggered this function removes any translated attributes, event handlers and context buttons.
//
//	 
//
// ARGUMENTS:
//   None
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.onRemoved = function()
{
  this.ensureValidElements();
  
  var contentElements = this.getContentElements();
  if ( !contentElements )
    return;
    
  for (var i = 0; i < contentElements.length; i++)
  {
    var contentElement = contentElements[i];
    contentElement.removeTranslatedAttribute("style");
    contentElement.removeTranslatedAttribute("class");
	var labelElement = this.getLabelForContent(contentElement);
    this.removeContextButton(labelElement);
    this.attachPanelHandlers(labelElement);
  }
};


//--------------------------------------------------------------------
// FUNCTION:
//   ensureValidDefaultPanel
//
// DESCRIPTION:
//   This function ensures that the default panel index refers to
//   an existing panel within the widget, or sets default index to -1 if the
//   widget has no panels.
//
// ARGUMENTS:
//   None
//
// RETURNS:
//   Index (integer) of the default panel, or -1 if no panel exists.
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.ensureValidDefaultPanel = function()
{
	if( this.defaultPanelId < 0 )
		this.defaultPanelId = 0;

  var contentElements = this.getContentElements();
	if( this.defaultPanelId >= contentElements.length )
		this.defaultPanelId = contentElements.length - 1;
	
  if ( this.isValidContent(contentElements[this.defaultPanelId],this.defaultPanelId) )
    return; //default is good already
    
  for ( var i = 0 ; i < contentElements.length ; i++ )
  {
    if ( this.isValidContent(contentElements[i], i) )
    {
      this.defaultPanelId = i;
      return;
    }
  }
  
  //no good panels
  if( dw.isDebugBuild() )
	alert("No valid panel found. Default panel not set.");
	
  this.defaultPanelId = -1;
};

//--------------------------------------------------------------------
// FUNCTION:
//   addNewPanel
//
// DESCRIPTION:
//   Adds a new panel after the open (current) panel.
//
// ARGUMENTS:
//  
// RETURNS:
//   Returns the DOM element object that represents the new panel
//   or undefined if a new panel was not added.
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.addNewPanel = function()
{
  var counter = 1;
  var contentElements = this.getContentElements();
  if ( contentElements && contentElements.length )
    counter = contentElements.length + 1;
  var panelSnippet = JQuery.DesignTime.Widget.Accordion.getNewPanelSnippet(counter);
  
  if ( this.currentContent && this.currentContent.ownerDocument )
  {
    // Add it after the current panel.
    this.currentContent.outerHTML = this.currentContent.outerHTML + panelSnippet
  }
  else if ( this.element && this.element.ownerDocument )
  {
    // Add it at the top.
    this.element.innerHTML = panelSnippet + this.element.innerHTML;
  }
};

//--------------------------------------------------------------------
// FUNCTION:
//   attachPanelHandlers
//
// DESCRIPTION:
//   Internal function that adds event handlers to the elements that
//   make up a single panel. These event handlers are meant to be
//   triggered from events generated by the user when interacting with
//   the widget in design view.
//
// ARGUMENTS:
//   panel - object - The DOM element that represents the panel
//                    within the widget's markup structure.
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.attachPanelHandlers = function(labelElement)
{

  if (!labelElement)
    return;

	var self = this;
	
    var canFireEvent = function()
    {
	if ( !self ) {
         return false;
      }
       
      self.ensureValidElements();
      
      if ( !self.currentContent || !self.currentContent.ownerDocument ) {
        self.refresh();
      }
      
      if ( !labelElement || !labelElement.ownerDocument ) {
	  
        return false;
      }
      return true;
    };
    
	var contentElement = this.getContentForLabel(labelElement);
    this.addEventListener(labelElement, "DWContextButtonActivate", function(e) { if ( canFireEvent() )  return self.onDWContextButtonActivate(contentElement); }, false);
  
};

//--------------------------------------------------------------------
// FUNCTION:
//   getLabelElements
//
// DESCRIPTION:
//   This method with return the list of all label node elements (Not the ids)
//
// ARGUMENTS:
//
// RETURNS:
//   Array of all Label Elements (or Section headings)
//--------------------------------------------------------------------
JQuery.DesignTime.Widget.Accordion.prototype.getLabelElements = function()
{
	this.ensureValidElements();
	var labelElements = new Array();
	var accordionChildren = this.getElementChildren(this.element);
	for (var i = 0; i < accordionChildren.length; i++)
	{
		if( i%2 == 0 )
		{
			labelElements.push(accordionChildren[i]);
		}
	}
	return labelElements;
}

//--------------------------------------------------------------------
// FUNCTION:
//   getContentElements
//
// DESCRIPTION:
//   This method with return the list of all content node elements (Sometimes used as 'panels')
//
// ARGUMENTS:
//
// RETURNS:
//   Array of all Content Elements 
//--------------------------------------------------------------------
JQuery.DesignTime.Widget.Accordion.prototype.getContentElements = function()
{
	this.ensureValidElements();
	var contentElements = new Array();
	var accordionChildren = this.getElementChildren(this.element);
	for (var i = 0; i < accordionChildren.length; i++)
	{
		
		if( i%2 == 1 )
		{
			contentElements.push(accordionChildren[i]);
		}
	}
	
	return contentElements;
}


//--------------------------------------------------------------------
// FUNCTION:
//   attachBehaviors
//
// DESCRIPTION:
//   Internal utility function for attaching event handlers to all
//   panels and setting the initial display state of the widget.
//
// ARGUMENTS:
//   openPanelIndex - integer - The index of the panel that should
//                              be open after this function has
//                              finished executing.
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.attachBehaviors = function(openPanelIndex)
{ 
  if( !this.isWidgetStructureValid() )
	return;
	
  var contentElements = this.getContentElements();
  for (var i = 0; i < contentElements.length; i++)
  {
    var content = contentElements[i];
	var labelElement = this.getLabelForContent(contentElements[i]);
    if (i == openPanelIndex)
    {
      this.currentContent = contentElements[i];
      this.removeClassName(this.getLabelForContent(this.currentContent), this.closedClass);
      this.addClassName(this.getLabelForContent(this.currentContent), this.openClass);
      if (content)
        JQuery.DesignTime.Widget.Accordion.setDisplay(content, "block");
    }
    else
    {
      this.removeClassName(this.getLabelForContent(contentElements[i]), this.openClass);
      this.addClassName(this.getLabelForContent(contentElements[i]), this.closedClass);
      if (content)
        JQuery.DesignTime.Widget.Accordion.setDisplay(content, "none");
      
	  // For the closed panels, add the open panel icon.
      if ( this.isValidContent(contentElements[i], i)) { 
        this.addShowPanelContextButton(labelElement);
      }
    }
    this.attachPanelHandlers(labelElement);
  }
};

//--------------------------------------------------------------------
// FUNCTION:
//   setDisplay
//
// DESCRIPTION:
//   This static method is used for setting the translated display
//   property of a DOM element. It uses internal methods for removing or showing
//	 panels in design view.
//
// ARGUMENTS:
//   ele - object - DOM element.
//   display - string - CSS value to set the display property to.
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.setDisplay = function(ele, display)
{
  if ( ele )
    ele.translatedStyle.display = display;
};

//--------------------------------------------------------------------
// FUNCTION:
//   onDWContextButtonActivate
//
// DESCRIPTION:
//   Internal event callback function for handling the click on the
//   activate button that appears in the tab of the panel in design view.
//   Clicking on this button in design view causes the corresponding
//   panel to open.
//
// ARGUMENTS:
//   contentElement - object - The DOM element that represents a content Element
//                    within the widget's markup structure.
//	 				
//					This is the content Element that opens after method execution 
//	 				(Most of these are design view only methods)
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.onDWContextButtonActivate = function(contentElement)
{
	dw.logEvent(UT_JQUERY_ACCORDION, UT_JQUERY_ACCORDION_OPEN_PANEL);
	
  if (contentElement != this.currentContent)
    this.openPanel(contentElement);
};

//--------------------------------------------------------------------
// FUNCTION:
//   JQuery.DesignTime.Widget.Accordion.getNewPanelSnippet
//
// DESCRIPTION:
//   Static utility function that returns a string containing the
//   markup for a new Accordion panel.
//
// ARGUMENTS:
//  counter - integer - Unique integer value appended to the tab label and
//                      content so that it is unique.
//
// RETURNS:
//   String that is the HTML markup fragment for the panel.
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.getNewPanelSnippet = function(counter)
{
  if ( typeof counter == 'number' ) counter = " " + counter;
  if ( typeof counter == 'undefined' ) counter = "";
  
  tab = dw.loadString('jquery/widgets/accordion/newSnippet/label'); 
  content = dw.loadString('jquery/widgets/accordion/newSnippet/content');
  
  return '<h3><a href="#">' + tab + counter + '</a></h3><div><p>' + content + counter + '</p></div>';
  /*<h3><a href="#">Section 1</a></h3>
	<div>
		<p>Content for section 1</p>
	</div>*/
};

//--------------------------------------------------------------------
// FUNCTION:
//   JQuery.DesignTime.Widget.Accordion.getNewAccordionSnippet
//
// DESCRIPTION:
//   Static utility function that returns a string containing the
//   markup for a complete Accordion.
//
// ARGUMENTS:
//  id - string - The string to use as the widget's id attribute.
//
// RETURNS:
//   String that is the HTML markup fragment for the panel.
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.getNewAccordionSnippet = function(id)
{
  var numPanels = 3;
  
  var accSnippet = '<div id="' + id + '">'
  for ( var i = 0 ; i < numPanels ; i++ ) {
    accSnippet += JQuery.DesignTime.Widget.Accordion.getNewPanelSnippet(i+1) ;
  }
  accSnippet += '</div>';
  
  return accSnippet;
};

//--------------------------------------------------------------------
// FUNCTION:
//   JQuery.DesignTime.Widget.Accordion.getNewAccordionConstructorSnippet
//
// DESCRIPTION:
//   Static utility function that returns a string that contains the
//   constructor snippet for creating a JQuery widget with the specified id.
//
// ARGUMENTS:
//  id - string - The id of the widget markup to insert into the
//                constructor snippet.
//
// RETURNS:
//   String containing JS that is the constructor call to create the
//   widget.
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.getNewAccordionConstructorSnippet = function(id)
{
  return ' $(function() {\n\t$( "#' + id + '" ).accordion(); \n});';
 /* $(function() {
		$( "#accordion" ).accordion();
	});*/
};

//--------------------------------------------------------------------
// FUNCTION:
//   isValidContent
//
// DESCRIPTION:
//   Checks the content element for being a valid 'div' element.
//	 Also checks the position of the content element in the Accordion
//	 Returns true only if both of the above conditions say this is the content element we mean to use
//
// ARGUMENTS:
//   contentElement - Object - The content node that we mean to validate
//	 index - Number - The expected index of the content node in the Accordion (not counting labels)
//
// RETURNS:
//   True if the content node is a valid 'div' node at the correct 'index'
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.isValidContent = function(contentElement, index)
{
	return (this.element.getElementsByTagName('div')[index] == contentElement);
}

//--------------------------------------------------------------------
// FUNCTION:
//   isValidLabel
//
// DESCRIPTION:
//   Checks the Label element for being a valid 'h3' element.
//	 Also checks the position of the Label element in the Accordion
//	 Returns true only if both of the above conditions say this is the Label element we mean to use
//
// ARGUMENTS:
//   labelElement - Object - The label node that we mean to validate
//	 index - Number - The expected index of the label node in the Accordion (not counting content Nodes or visible panels)
//
// RETURNS:
//   True if the label node is a valid 'h3' node at the correct 'index'
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.isValidLabel = function(labelElement ,index)
{
	return (this.element.getElementsByTagName('h3')[index] == labelElement);
}

//--------------------------------------------------------------------
// FUNCTION:
//   JQuery.DesignTime.Widget.Accordion.areConsecutiveLabelPanel
//
// DESCRIPTION:
//   This static method checks whther the given label and content nodes are siblings (in the correct order)
//
// ARGUMENTS:
//   labelElement - 	Object - The label node 
//	 contentElement - 	Object - The content node directly after the label node
//
// RETURNS:
//   True if the label node is an immediately previous sibling of the contentNode
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.areConsecutiveLabelPanel = function(labelElement , contentElement)
{
	var panelForLabel = labelElement.nextSibling;
	
	return panelForLabel == contentElement;	
}

//--------------------------------------------------------------------
// FUNCTION:
//   getLabelForContent
//
// DESCRIPTION:
//   Fetches the matching label node for a content node
//
// ARGUMENTS:
//	 contentElement - 	Object - The content node for which we want the corresponding label node
//
// RETURNS:
//   The label node for this content
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.getLabelForContent = function (contentElement)
{
	if(!contentElement)
		return null;
	return contentElement.previousSibling;	
}

//--------------------------------------------------------------------
// FUNCTION:
//   getContentForLabel
//
// DESCRIPTION:
//   Fetches the matching content node for a label node
//
// ARGUMENTS:
//	 labelElement - 	Object - The label node for which we want the corresponding content node
//
// RETURNS:
//   The content node for this label
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.getContentForLabel = function (labelElement)
{
	if(!labelElement)
		return null;
	return labelElement.nextSibling;	
}

//--------------------------------------------------------------------
// FUNCTION:
//   isWidgetStructureValid
//
// DESCRIPTION:
//   The method valides the complete structure of an accordion, checkin for 'h3' 
//	 and 'div' nodes in the right order and number.
//
// ARGUMENTS:
//
// RETURNS:
//   True if the Accordion is a valid a JQueryUI Accordion
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.isWidgetStructureValid = function ()
{
	var accordionChildren = this.getElementChildren(this.element);
	
	if( !accordionChildren )
		return;
			
	for( var i = 0; i < accordionChildren.length; i++ )
	{
		if( i%2 == 0 )				//labelElement
		{
			if( !this.isValidLabel(accordionChildren[i], Math.floor(i/2)))
				return false;
		}
		else						//should be a valid content element
		{
			if( !this.isValidContent(accordionChildren[i], Math.floor(i/2)) )
				return false;
		}
	}
	
	var labelElements = this.getLabelElements();
	var contentElements = this.getContentElements();
	
	if( labelElements.length != contentElements.length )
		return false;
		
	for( var i = 0; i < labelElements.length; i++ )
	{	
		if( !JQuery.DesignTime.Widget.Accordion.areConsecutiveLabelPanel(labelElements[i], contentElements[i]) )
			return false;
	}
	return true;					//all is well!
}

JQuery.DesignTime.Widget.Accordion.prototype.getActivePanel = function()
{
  return (this.opts && this.opts.active && typeof this.opts.active != 'undefined')? this.opts.active : 0;
};
JQuery.DesignTime.Widget.Accordion.prototype.getCollapsibleValue = function()
{
  if( this.opts && this.opts.collapsible != null && typeof this.opts.collapsible != 'undefined' )
	{
		return (this.opts.collapsible == true);
	}
	return false;
};
JQuery.DesignTime.Widget.Accordion.prototype.getEventIndex = function()
{
  if( this.opts && this.opts.event != null && typeof this.opts.event != 'undefined' )
	{
		var eventArray = this.getEventOptions();
		for( var i = 0; i < eventArray.length; i++ )
		{
			if( this.opts.event == eventArray[i] )
				return i;
		}
		return 0;
	}
	return 0;
};
JQuery.DesignTime.Widget.Accordion.prototype.setActiveValue = function(activeValue)
{
	if( activeValue == null || typeof activeValue == "undefined" )
		return;
		
	if (!this.opts)
		this.opts = {};
	
	var oldActiveValue = this.opts.active;
	
	if( activeValue == "" || activeValue.toString() == '0' || parseInt(activeValue).toString() == 'NaN' )
	{
		this.opts.active = null;
	}	
	else
	{
		this.opts.active = parseInt(activeValue);
	}
	
	if( oldActiveValue != this.opts.active )
		this.updateOptions(this.widgetType, this.opts);
};
JQuery.DesignTime.Widget.Accordion.prototype.setCollapsibleValue = function(collapsibleValue)
{
	if( collapsibleValue == null || typeof collapsibleValue == "undefined" )
		return;

	if (!this.opts)
		this.opts = {};
		
	if( collapsibleValue == true )
	{
		this.opts.collapsible = true;
	}	
	else
	{
		this.opts.collapsible = null;
	}
	
	this.updateOptions(this.widgetType, this.opts);
};
JQuery.DesignTime.Widget.Accordion.prototype.setEvent = function(eventIndex)
{
	if (!this.opts)
		this.opts = {};
	
	var oldEventValue = this.opts.event;
	
	if( eventIndex == 0 )
		this.opts.event = null;
	else
		this.opts.event = this.getEventOptions()[eventIndex];
		
	if( this.opts.event != oldEventValue )	
		this.updateOptions(this.widgetType, this.opts);
};
JQuery.DesignTime.Widget.Accordion.prototype.initEventArray = function()
{
	this.eventArray.push("click");
	this.eventArray.push("mouseover");
}
JQuery.DesignTime.Widget.Accordion.prototype.initHeightStyleArray = function()
{
	this.heightStyleArray.push("auto");
	this.heightStyleArray.push("fill");
	this.heightStyleArray.push("content");
}
JQuery.DesignTime.Widget.Accordion.prototype.initEasingArray = function()
{
	this.easingArray.push("swing");
	this.easingArray.push("linear");
	this.easingArray.push("easeInQuad");
	this.easingArray.push("easeOutQuad");
	this.easingArray.push("easeInOutQuad");
	this.easingArray.push("easeInCubic");
	this.easingArray.push("easeOutCubic");
	this.easingArray.push("easeInOutCubic");
	this.easingArray.push("easeInQuart");
	this.easingArray.push("easeOutQuart");
	this.easingArray.push("easeInOutQuart");
	this.easingArray.push("easeInQuint");
	this.easingArray.push("easeOutQuint");
	this.easingArray.push("easeInOutQuint");
	this.easingArray.push("easeInExpo");
	this.easingArray.push("easeOutExpo");
	this.easingArray.push("easeInOutExpo");
	this.easingArray.push("easeInSine");
	this.easingArray.push("easeOutSine");
	this.easingArray.push("easeInOutSine");
	this.easingArray.push("easeInCirc");
	this.easingArray.push("easeOutCirc");
	this.easingArray.push("easeInOutCirc");
	this.easingArray.push("easeInElastic");
	this.easingArray.push("easeOutElastic");
	this.easingArray.push("easeInOutElastic");
	this.easingArray.push("easeInBack");
	this.easingArray.push("easeOutBack");
	this.easingArray.push("easeInOutBack");
	this.easingArray.push("easeInBounce");
	this.easingArray.push("easeOutBounce");
	this.easingArray.push("easeInOutBounce");
}
JQuery.DesignTime.Widget.Accordion.prototype.initHeaderArray = function()
{
	this.headerArray.push("ui-icon-triangle-1-e");
	this.headerArray.push("ui-icon-carat-1-e");
	this.headerArray.push("ui-icon-arrow-1-e");
	this.headerArray.push("ui-icon-arrowthick-1-e");
	this.headerArray.push("ui-icon-plus");
	this.headerArray.push("ui-icon-plusthick");
	this.headerArray.push("ui-icon-circle-plus");
	this.headerArray.push("ui-icon-circle-triangle-e");
	this.headerArray.push("ui-icon-circle-arrow-e");
	this.headerArray.push("ui-icon-circlesmall-plus");
	this.headerArray.push("ui-icon-squaresmall-plus");
}
JQuery.DesignTime.Widget.Accordion.prototype.initActiveHeaderArray = function()
{
	this.activeHeaderArray.push("ui-icon-triangle-1-s");
	this.activeHeaderArray.push("ui-icon-carat-1-s");
	this.activeHeaderArray.push("ui-icon-arrow-1-s");
	this.activeHeaderArray.push("ui-icon-arrowthick-1-s");
	this.activeHeaderArray.push("ui-icon-minus");
	this.activeHeaderArray.push("ui-icon-minusthick");
	this.activeHeaderArray.push("ui-icon-circle-triangle-s");
	this.activeHeaderArray.push("ui-icon-circle-arrow-s");
	this.activeHeaderArray.push("ui-icon-circlesmall-minus");
	this.activeHeaderArray.push("ui-icon-squaresmall-minus");
}
JQuery.DesignTime.Widget.Accordion.prototype.getEventOptions = function ()
{
	return this.eventArray;
}
JQuery.DesignTime.Widget.Accordion.prototype.getHeightStyleOptions = function ()
{
	return this.heightStyleArray;
}
JQuery.DesignTime.Widget.Accordion.prototype.getEasingOptions = function ()
{
	return this.easingArray;
}
JQuery.DesignTime.Widget.Accordion.prototype.getHeaderOptions = function ()
{
	return this.headerArray;
}
JQuery.DesignTime.Widget.Accordion.prototype.getActiveHeaderOptions = function ()
{
	return this.activeHeaderArray;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getHeightStyleIndex
//
// DESCRIPTION:
//  	This method will return the index of the heightStyle List element
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Number: The value of the index of dateformat list
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.getHeightStyleIndex = function()
{
	if( this.opts && this.opts.heightStyle != null && typeof this.opts.heightStyle != 'undefined' )
	{
		var heightStyleArray = this.getHeightStyleOptions();
		for( var i = 0; i < heightStyleArray.length; i++ )
		{
			if( this.opts.heightStyle.toString() == heightStyleArray[i] )
				return i;
		}
	}
	return 0;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setHeightStyleIndex
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the heightStyle
//		list.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.setHeightStyleIndex = function(heightStyleIndex)
{
	if( heightStyleIndex == null || typeof heightStyleIndex == "undefined" )
		return;
		
	if (!this.opts)
		this.opts = {};
	
	var oldDateValue = this.opts.heightStyle;
	
	if( heightStyleIndex == 0 )
		this.opts.heightStyle = null;
	else
		this.opts.heightStyle = this.getHeightStyleOptions()[heightStyleIndex];
	
	if( oldDateValue != this.opts.heightStyle )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getHeaderIndex
//
// DESCRIPTION:
//  	This method will return the index of the header class values list that should be displayed in
//		the PI. In case user has entered a new value in code himself, it may also
//		return the value of the newly entered number for header
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Number: The value of the index of primary icon list or a new string value for the icon class
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.getHeaderIndex = function()
{
	if( this.opts && this.opts.icons != null && typeof this.opts.icons != 'undefined' )
	{
		if( this.opts.icons.header && typeof this.opts.icons.header != 'undefined')
		{
			var headerArray = this.getHeaderOptions();
			for( var i = 0; i < headerArray.length; i++ )
			{
				if( this.opts.icons.header.toString() == headerArray[i] )
					return i;
			}
			if( this.opts.icons.header.toString() != "" )
			{
				headerArray.push(this.opts.icons.header.toString());
				return headerArray.length-1;
			}
		}
	}
	return 0;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setHeaderIndex
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the header list
//		list.
//
// ARGUMENTS:
//  	index of primary icon list
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.setHeaderIndex = function(listIndex)
{
	if( listIndex == null || typeof listIndex == "undefined" )
		return;
			
	if (!this.opts)
		this.opts = {};
		
	if (!this.opts.icons)
		this.opts.icons = {};
	
	var oldHeaderValue = this.opts.icons.header;
	var headerArray = this.getHeaderOptions();
	
	if( listIndex == 0 )
		this.opts.icons.header = null;
	else
		this.opts.icons.header = headerArray[listIndex];	
		
	if( this.opts.icons && !this.opts.icons.header && !this.opts.icons.activeHeader )
	{
		this.opts.icons = null;	
		this.updateOptions(this.widgetType, this.opts);	
	}	
	else if( this.opts.icons.header != oldHeaderValue )	
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getActiveHeaderIndex
//
// DESCRIPTION:
//  	This method will return the index of the secondary list that should be displayed in
//		the PI. In case user has entered a new value in code himself, it may also
//		return the value of the newly entered number for active header
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Number: The value of the index of secondary icon list or a new string value for the icon class
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.getActiveHeaderIndex = function()
{
	if( this.opts && this.opts.icons != null && typeof this.opts.icons != 'undefined' )
	{
		if( this.opts.icons.activeHeader && typeof this.opts.icons.activeHeader != 'undefined')
		{
			var activeHeaderArray = this.getActiveHeaderOptions();
			for( var i = 0; i < activeHeaderArray.length; i++ )
			{
				if( this.opts.icons.activeHeader.toString() == activeHeaderArray[i] )
					return i;
			}
			if( this.opts.icons.activeHeader.toString() != "" )
			{
				activeHeaderArray.push(this.opts.icons.activeHeader.toString());
				return activeHeaderArray.length-1;
			}
		}
	}
	return 0;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setActiveHeaderIndex
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the secondary icon
//		list.
//
// ARGUMENTS:
//  	index of secondary icon list
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.setActiveHeaderIndex = function(listIndex)
{
	if( listIndex == null || typeof listIndex == "undefined" )
		return;
			
	if (!this.opts)
		this.opts = {};
		
	if (!this.opts.icons)
		this.opts.icons = {};
	
	var oldActiveHeaderValue = this.opts.icons.activeHeader;
	var activeHeaderArray = this.getActiveHeaderOptions();
	
	if( listIndex == 0 || listIndex >= activeHeaderArray.length )
		this.opts.icons.activeHeader = null;
	else
		this.opts.icons.activeHeader = activeHeaderArray[listIndex];	
	
	if( this.opts.icons && !this.opts.icons.header && !this.opts.icons.activeHeader )
	{
		this.opts.icons = null;
		this.updateOptions(this.widgetType, this.opts);	
	}	
	else if( this.opts.icons.activeHeader != oldActiveHeaderValue )	
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getEasingIndex
//
// DESCRIPTION:
//  	This method will return the index of the easing list that should be displayed in
//		the PI. In case user has entered a new value in code himself, it may also
//		return the value of the newly entered number for easing
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Number: The value of the index of list or a new string value for the icon class
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.getEasingIndex = function()
{
	if( this.opts && this.opts.animate != null && typeof this.opts.animate != 'undefined' )
	{
		if( this.opts.animate.easing && typeof this.opts.animate.easing != 'undefined')
		{
			var easingArray = this.getEasingOptions();
			for( var i = 0; i < easingArray.length; i++ )
			{
				if( this.opts.animate.easing.toString() == easingArray[i] )
					return i;
			}
			if( this.opts.animate.easing.toString() != "" )
			{
				easingArray.push(this.opts.animate.easing.toString());
				return easingArray.length-1;
			}
		}
	}
	return 0;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setEasingIndex
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the easing 
//		list.
//
// ARGUMENTS:
//  	index of secondary icon list
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.setEasingIndex = function(listIndex)
{
	if( listIndex == null || typeof listIndex == "undefined" )
		return;
			
	if (!this.opts)
		this.opts = {};
		
	if (!this.opts.animate)
		this.opts.animate = {};
	
	var oldEasingValue = this.opts.animate.easing;
	var easingArray = this.getEasingOptions();
	
	if( listIndex == 0 || listIndex >= easingArray.length )
		this.opts.animate.easing = null;
	else
		this.opts.animate.easing = easingArray[listIndex];	
	
	if( this.opts.animate && !this.opts.animate.duration && !this.opts.animate.easing )
	{
		this.opts.animate = null;
		this.updateOptions(this.widgetType, this.opts);	
	}	
	else if( this.opts.animate.easing != oldEasingValue )	
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setAnimateEasingInput
//
// DESCRIPTION:
//  	Here we set the opts object from the new value that user has 'typed' 
//		in the select list. We will not find this value in our list of values
//		It must be entered seperately
//
// ARGUMENTS:
//  	New value of Animate easing class
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.setAnimateEasingInput = function(easingValue)
{
	if( easingValue == null || typeof easingValue == "undefined" )
		return;
			
	if (!this.opts)
		this.opts = {};
		
	if (!this.opts.animate)
		this.opts.animate = {};
	
	var oldEasingValue = this.opts.animate.easing;
	
	if( easingValue == "" || easingValue == "none" )
		this.opts.animate.easing = null;
	else
		this.opts.animate.easing = easingValue;	
		
	if( this.opts.animate && !this.opts.animate.easing && !this.opts.animate.duration )
	{
		this.opts.animate = null;	
		this.updateOptions(this.widgetType, this.opts);	
	}	
	else if( this.opts.animate.easing != oldEasingValue )	
		this.updateOptions(this.widgetType, this.opts);	
}


//--------------------------------------------------------------------
// FUNCTION:
//  	getDurationValue
//
// DESCRIPTION:
//  	This method will return the value of duration that should be displayed in
//		the PI. In case user has entered a new value in code himself, it may also
//		return the value of the newly entered number for duration
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Number: The value of the index of list or a new string value for the icon class
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.getAnimateDurationValue = function()
{
	if( this.opts && this.opts.animate != null && typeof this.opts.animate != 'undefined' )
	{
		if( this.opts.animate.duration && typeof this.opts.animate.duration != 'undefined')
		{
			return this.opts.animate.duration;
		}
	}
	return 400;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setDurationIndex
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the duration 
//		list.
//
// ARGUMENTS:
//  	index of secondary icon list
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.setAnimateDurationValue = function(durationValue)
{
	if( durationValue == null || typeof durationValue == "undefined" )
		return;
		
	if (!this.opts)
		this.opts = {};
		
	if (!this.opts.animate)
		this.opts.animate = {};	
	
	var oldDurationValue = this.opts.animate.duration;
	
	if( durationValue == "" || durationValue == "400" || parseInt(durationValue).toString() == 'NaN' )
	{
		this.opts.animate.duration = null;
	}	
	else
	{
		this.opts.animate.duration = parseInt(durationValue);
	}
	
	if( this.opts.animate && !this.opts.animate.easing && !this.opts.animate.duration )
	{
		this.opts.animate = null;	
		this.updateOptions(this.widgetType, this.opts);	
	}	
	else if( this.opts.animate.duration != oldDurationValue )	
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getDisabledValue
//
// DESCRIPTION:
//  	This method returns the value of the disabled attribute in the widget constructor.
//		If user changes the disabled value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Boolean: The value of the disabled attribute as boolean
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.getDisabledValue = function()
{	
	if( this.opts && this.opts.disabled != null && typeof this.opts.disabled != 'undefined' )
	{
		return (this.opts.disabled == true);
	}
	return false;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setDisabledValue
//
// DESCRIPTION:
//  	When a new value for disabled is attained for the disabled attribute from the PI
//		we have to update our opts object with this new disabled value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Accordion.prototype.setDisabledValue = function(disabledValue)
{	
	if( disabledValue == null || typeof disabledValue == "undefined" )
		return;

	if (!this.opts)
		this.opts = {};
		
	var oldDisabledValue = this.opts.disabled	;
	if( disabledValue == true )
	{
		this.opts.disabled = true;
	}	
	else
	{
		this.opts.disabled = null;
	}
	
	if( oldDisabledValue != this.opts.disabled )
	{
		this.updateOptions(this.widgetType, this.opts);
	}
}



