/*	
 Copyright 2014 Adobe Systems Incorporated.  All rights reserved. 

Purpose- 
This file has the implementation of infrastructure needed for liveView extensions
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50, continue: true */
/*global window, DW_EVENTS, DW_LIVEEDIT_CONSTANTS, CustomEvent, DW_EXTENSION_EVENT, DW_PARENT_EVENT, DW_EDITABLE_SELECTORS_CONSTS, DW_AUX_HUD_CONSTS, Node , gFluidGridData, gFluidGridUrl, ResponsiveFrameworkLookup, ResponsiveLayoutConstants, $dwJQuery, DW_EVENTS_PARAM, DOMParser, RESPONSIVE_FRAMEWORK, CSSRule */

/*
DWObject - exposes functions which need to access DW (like querying and updating document)
*/
(function () {
    'use strict';
    
    function DWObjectForExtensions() {
    }

    DWObjectForExtensions.prototype.getAppLanguage = function () {
        return window.appLanguage;
    };

    /**
        Function: setResponsiveFeatures - Sets the list of responsive features present for the responsive framework
        Argument: frameworkName, [] of features
        Return Object : None
    */
    DWObjectForExtensions.prototype.setResponsiveFeatures = function (frameworkName, supportedFeatures, isFrameworkDetectionComplete) {
        var argsObj = {
            "frameworkName": frameworkName,
            "supportedFeatures": supportedFeatures,
            "isFrameworkDetectionComplete": isFrameworkDetectionComplete
        };
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'setResponsiveFeatures', argsObj);
    };
    
		/**
        Function: shouldReloadBootstrapFramework - returns true if we need to reload the bootstrapframework
        Argument: callbackFunc
        Return Object : None
    */
    DWObjectForExtensions.prototype.shouldReloadBootstrapFramework = function (callbackFunc) { 
        var argsObj = {
            callback: callbackFunc
        };
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'shouldReloadBootstrapFramework', argsObj);
    };
	
	/**
        Function: setReloadBootstrapFramework - sets a flag on document object, if we need to reload bootstrapframework
        Argument: a boolean flag to reset ReloadBootstrapFramework flag
        Return Object : None
    */
    DWObjectForExtensions.prototype.setReloadBootstrapFramework = function (shouldReload) {
        var argsObj = {
            "reloadBootstrapFramework": shouldReload
        };
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'setReloadBootstrapFramework', argsObj);
    };
	
    DWObjectForExtensions.prototype.readFile = function (fileName, callbackfunc) {
        var argObj = {};
        argObj.fileName = fileName;
        argObj.callback = callbackfunc;
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'readFile', argObj);
    };

    /**
        Function: isInUnhideMode - Fetches the state of unhide mode from the current TitanDoc
        Argument: callback function to be called with the result
        Return Object : None
    */
    DWObjectForExtensions.prototype.isInUnhideMode = function (callbackFunc) {
        var argsObj = {
            callback: callbackFunc
        };
        
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'isInUnhideMode', argsObj);
    };
    
     /*isTemplate :-Return Whether a Document is template or not */

    DWObjectForExtensions.prototype.isTemplate = function (callbackFunc) {
        var argsObj = {
            callback: callbackFunc
        };
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'isTemplate',argsObj);
    };
    /*
    Function: getDWDocumentClassesAndIds - Gets the list of available classes and Ids for use in this document

    Argument: callbackfunc -> callback to be called after executing

    Return Object : 
    allSelectors - array of available selectors
    */
    DWObjectForExtensions.prototype.getDWDocumentClassesAndIds = function (callbackfunc) {
        var callbackObj = {};
        callbackObj.callback = callbackfunc;
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'getDWDocumentClassesAndIds', callbackObj, false);
    };
    
    /*
    Function: checkIfFluidElement
    Returns true if the selected element is Fluid in an FG page    
    */
    
    DWObjectForExtensions.prototype.checkIfFluidElement = function (argObj) {
        
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'checkIfFluidElement', argObj, false);
    };
    
    /*
    Function: isCCLibLinkedImage
    Returns: The state of Linked Asset if the selected img element is a linked asset from CC Library    
    */
    
    DWObjectForExtensions.prototype.isCCLibLinkedImage = function (argObj) {
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'isCCLibLinkedImage', argObj, false);
    };
    
    /*
    Function: isAssetOptInProgress
    Returns: This method returns if asset optimization is in process   
    */
    
    DWObjectForExtensions.prototype.isAssetOptInProgress = function (argObj) {
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'isAssetOptInProgress', argObj, false);
    };
    
    /*
    Function: handleCloudIconClick
    Argument is the image src of the linked asset from CCLibraries.
             Invoked on clicking the cloud icon from ESH.
    */
    
    DWObjectForExtensions.prototype.handleCloudIconClick = function (argObj) {
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'handleCloudIconClick', argObj, false);
    };
    
    /*
    Function: getDWDocumentElementClassAndID - Gets the class and id attributes from document for the passend element

    Argument:   elem - Corresponding live view element
                callbackfunc - callback to be called after executing

    Return Object : object.classStr - Value of class attribute
                    object.idStr    - Value of id attribute
    */
    DWObjectForExtensions.prototype.getDWDocumentElementClassAndID = function (elem, callbackfunc) {
        if (!elem) {
            return;
        }
        
        //get the unique ID for the element to pass to DW side
        var uniqId = elem.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
        
        var argObj = {};
        argObj.uniqId = uniqId;
        argObj.callback = callbackfunc;
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'getDWDocumentElementClassAndID', argObj, false);
    };

    /*
    function:updateDWDocElementAttr - update an attribute value in the DW document for the element

    Arguments:

        elem - Corresponding live view element
        attrName - what attribute needs to be changed
        attrValue - new value of the attribute

    Return Obj : Nothing
    */
    DWObjectForExtensions.prototype.updateDWDocumentElementAttr = function (elem, attrName, attrVal, callback, selId) {
        if (!elem || elem.nodeType !== Node.ELEMENT_NODE) {
            return;
        }

        //get the unique ID for the element to pass to DW side
        var uniqId = elem.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
        if (!uniqId || uniqId.length === 0) {
            return;
        }

        //propogate to DW
        var argObj = {};
        argObj.uniqId = uniqId;
        argObj.attrName	= attrName;
        argObj.attrValue = attrVal;
        if (callback) {
            argObj.callback = callback;
        }
		if (selId) {
			argObj.selectionId = selId;
		}
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'updateDWDocElementAttr', argObj, true);
    };
   
    DWObjectForExtensions.prototype.invokeNFWHint = function (argObj) {
        if (!argObj) {
            return;
        }
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, 'invokeNFWHint', argObj, false);
    };
    
    /**
     * function:dwBrowseForFileURL
     * (Internal function on spider monkey side)
     * @param argObj
     * operation: select/open
     * subop : name that will be shown at the top
     * fileter: filters to be applied while showing the folder 
     * callback: on successful/failure of execution callback to js layer.
     * Description: 
     * Browse for file functionality for source and link.
     */
    DWObjectForExtensions.prototype.dwBrowseForFileURL = function (operation, subOperation, callback, filter, stopEditOp) {
        if (!operation || !subOperation) {
            return;
        }

        //propogate to DW
        var argObj = {};
        argObj.filter = filter;
        argObj.operation = operation;
        argObj.subOp = subOperation;
        argObj.callback = callback;

        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, "dwBrowseForFileURL", argObj, stopEditOp);
    };
    
    /**
     * function:dwGetRelatedCSSFiles
     * (Internal function on spider monkey side)
     * @param argObj
     * callback: on successful/failure of execution callback to js layer.
     * Description: 
     * Get CSS files related to document.
     */
    DWObjectForExtensions.prototype.dwGetRelatedCSSFiles = function (argObj, stopEditOp) {
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.HudSelectorHud, "dwGetRelatedCSSFiles", argObj, stopEditOp);
    };
    
    /**
     * function:dwGetRelatedMediaQueries
     * (Internal function on spider monkey side)
     * @param argObj
     * callback: on successful/failure of execution callback to js layer.
     * Description: 
     * Get Media queries related to a CSS file.
     */
    DWObjectForExtensions.prototype.dwGetRelatedMediaQueries = function (argObj, stopEditOp) {
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.HudSelectorHud, "dwGetRelatedMediaQueries", argObj, stopEditOp);
    };
    
    /**
     * function:dwGetNewCSSDialog
     * (Internal function on spider monkey side)
     * @param argObj
     * callback: on successful/failure of execution callback to js layer.
     * Description: 
     * Launches new CSS file dialog.
     */
    DWObjectForExtensions.prototype.dwGetNewCSSDialog = function (argObj, stopEditOp) {
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.HudSelectorHud, "dwGetNewCSSDialog", argObj, stopEditOp);
    };
    
    /**
     * function:dwAddCssRule
     * (Internal function on spider monkey side)
     * @param argObj
     * callback: on successful/failure of execution callback to js layer.
     * Description: 
     * Add CSS rule to document.
     */
    DWObjectForExtensions.prototype.dwAddCssRule = function (argObj, stopEditOp) {
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.HudSelectorHud, "dwAddCssRule", argObj, stopEditOp);
    };
    /**
     * function:dwGetNewMeidaQueryDialog
     * (Internal function on spider monkey side)
     * @param argObj
     * callback: on successful/failure of execution callback to js layer.
     * Description: 
     * Add CSS rule to document.
     */
    DWObjectForExtensions.prototype.dwGetNewMeidaQueryDialog = function (argObj, stopEditOp) {
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.HudSelectorHud, "dwGetNewMeidaQueryDialog", argObj, stopEditOp);
    };
    
    /**
     * function:dwDefineCSSRuleInPage
     * (Internal function on spider monkey side)
     * @param argObj
     * callback: on successful/failure of execution callback to js layer.
     * Description: 
     * Define CSS rule in HTML document.
     */
    DWObjectForExtensions.prototype.dwDefineCSSRuleInPage = function (argObj, stopEditOp) {
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.HudSelectorHud, "dwDefineCSSRuleInPage", argObj, stopEditOp);
    };
    
    /*
    Function: DWSMSetSelectionOnNodeWithId
    Arguments: 
    id -> dwId
    */
    
    DWObjectForExtensions.prototype.DWSMSetSelectionOnNodeWithId = function (id, forceSyncSelection) {
        var argObj = {};
        argObj.id = id;
        argObj.forceSyncSelection = forceSyncSelection !== null ? forceSyncSelection : false;
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, "DWSMSetSelectionOnNodeWithId", argObj, false);
    };

    /*
    Function: DWSMSetSelectionOnNodeWithIdAndInsertHTML
    Arguments: 
                DWID: dwId,
                HTML: htmlToBeInserted,
                REPLACE_SELECTION: false,
                SWITCH_SELECTION: true
    */
    DWObjectForExtensions.prototype.DWSMSetSelectionOnNodeWithIdAndInsertHTML = function (argsObj, stopEditOp) {
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, "DWSMSetSelectionOnNodeWithIdAndInsertHTML", argsObj, stopEditOp);
    };
    /*
    Function: logHeadlightsData
    Arguments: 
    category -> feature area in which this event occurred
    eventString -> event occurred
    */
    
    DWObjectForExtensions.prototype.logHeadlightsData = function (subCategory, eventString, category) {
        var argObj = {};
        argObj.subCategory = subCategory;
        argObj.eventString = eventString;
        if (category !== undefined) {
            /* This is the case when you need Headlights to be logged under a category other than 'DW_Usage_Data'.
            */
            argObj.category = category;
        }
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, "DWLogHeadlightsData", argObj, false);
    };
    
    /*
    function:DWSMGetDWIdsForUpdatedTags
    (Internal function on spider monkey side)
    Arguments:argObj
        
        nodeId      - dwid of node under edit
        parentId    - dwid of ParentNode of that node
        newText     - new text to keep in the place of old content at that offset.
        counter     - how many nodes have been modified(to get corresponding node dwid's for tempid's)
        callback    - callback function to be called
    
    Description: 
        Update the link attribute and fetch dwid's for all the modfied nodes.
    
    */
    DWObjectForExtensions.prototype.DWSMGetDWIdsForUpdatedTags = function (argObj) {
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, "DWSMGetDWIdsForUpdatedTags", argObj, true);
    };

    /**
     * function:dwHandleLiveViewDuplicate
     * (Internal function on spider monkey side)
     * @param : none
     * Description: 
     * Duplicate the layout and its content of selected element in Responsive framework layout file
     */
    DWObjectForExtensions.prototype.dwHandleLiveViewDuplicate = function () {
        var argObj = {};
        window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, "dwHandleLiveViewDuplicate", argObj);
    };
    
    /*
    Live view object - Used for the loading and handling of extensions in LiveView
    */
    function LiveViewObjectForExtensions() {
        this.currentSelectedElement = null;
        this.isANonEditableElement = false;
        this.liveViewExtensionsArray = [];
		// Hard coding the help ids for now. Will change it once we have the proper mapping.
		this.liveViewExtensionsArrayForHelp = [{extensionId: "FluidGrid", helpId: "AH_FLUIDGRID"}, {extensionId: "dwImageHud", helpId: "AH_IMAGE_HUD"}, {extensionId: "dwTextHud", helpId: "AH_TEXT_HUD_QPI"}, {extensionId: "editableSelectors", helpId: "AH_ESH"}];
        this.auxiliaryIframeExists = {};
        this.dwObject = new DWObjectForExtensions();
		this.liveExtensionsConfigPath = null;
        this.auxHudVisbility = {};
        this.currentDraggedElement = null;
        this.initializeResponsiveFrameworkConstants();
        this.liveViewResizing = false;
        this.currentTheme = window.DWCurrentTheme || "";
        this.liveEditInProgress = false;
        this.liveEditCommitPending = false;
        this.dwIdForSelectChildEvent = null;
    }

    /*
    function:getDwIdForSelectChildEvent - get the dwId for the node in Live View for 
                                          select_child_in_liveview event.

    Arguments: none

    Return: Returns the dwId which is set
            using setDwIdForSelectChildEvent function
    */
    LiveViewObjectForExtensions.prototype.getDwIdForSelectChildEvent = function () {
        return this.dwIdForSelectChildEvent;
    };

    /*
    function:setDwIdForSelectChildEvent - set the dwId for the node in Live View for 
                                          select_child_in_liveview event.

    Arguments: dwId - dwID of the node which is to be set

    Return: none
    */
    LiveViewObjectForExtensions.prototype.setDwIdForSelectChildEvent = function (dwId) {
        this.dwIdForSelectChildEvent = dwId;
    };

    /*
    function:getCurrentSelectedElement - get the current selected node in Live View. This is the live view 
                                         element corresponding to the selected element in DW DOM.

    Arguments: none

    Return: Returns the DOM element which is set
            using setCurrentSelectedElement function
    */
    LiveViewObjectForExtensions.prototype.getCurrentSelectedElement = function () {
        return this.currentSelectedElement;
    };
    
    /*
    function:getCurrentDraggedElement - get the current dragged node in Live View. This is the live view 
                                         element corresponding to the dragged element in DW DOM.

    Arguments: none

    Return: Returns the DOM element which is set
            using setCurrentDraggedElement function
    */
    LiveViewObjectForExtensions.prototype.getCurrentDraggedElement = function () {
        return this.currentDraggedElement;
    };
    
    /*
    function:setCurrentDraggedElement - set the current dragged node in Live View.

    Arguments: uniqueID - DW Id of the node being dragged

    Return: None
    */
    LiveViewObjectForExtensions.prototype.setCurrentDraggedElement = function (uniqNodeId) {
        var targetElement = null;
        var querySelectorString = "";
        if (uniqNodeId && uniqNodeId.length > 0) {
            //in case we have a dw node id, then use query selector to get the element with attribute
            querySelectorString = '[' + DW_LIVEEDIT_CONSTANTS.DWUniqueId + '="' + uniqNodeId + '"]';
        
            try {
                var elems = document.querySelectorAll(querySelectorString);
                if (elems && elems.length > 0) {
                    targetElement = elems[0];
                }
            } catch (e) {
            }
        }
                
        this.currentDraggedElement = targetElement;
    };
    
    LiveViewObjectForExtensions.prototype.setCurrentTheme = function (themeName) {
        this.currentTheme = themeName;
    };
    
    LiveViewObjectForExtensions.prototype.getCurrentTheme = function () {
        return this.currentTheme;
    };
    
    /*
    function:enableLiveDragFeedback - enable the Live Drag Feedback extension

    Arguments: none

    Return: None
    */
    LiveViewObjectForExtensions.prototype.enableLiveDragFeedback = function () {
        var extensionIndex = this.getExtensionIndexById(DW_LIVEEDIT_CONSTANTS.DwLiveDragFeedbackExtensionId);
        if (extensionIndex >= 0) {
            this.liveViewExtensionsArray[extensionIndex].enabled = true;
        }
    };

    LiveViewObjectForExtensions.prototype.setAuxHudVisibility = function (tag, extensionID, visibility) {
        if (!this.auxHudVisbility.hasOwnProperty(tag)) {
            this.auxHudVisbility[tag] = {};
        }
        
        this.auxHudVisbility[tag][extensionID] = visibility;
    };
    
    LiveViewObjectForExtensions.prototype.getAuxHudVisibility = function (tag, visibility) {
        if (this.auxHudVisbility.hasOwnProperty(tag)) {
            var auxHudVisibile = false,
                property;
            for (property in this.auxHudVisbility[tag]) {
                if (this.auxHudVisbility[tag].hasOwnProperty(property)) {
                    auxHudVisibile = auxHudVisibile || this.auxHudVisbility[tag][property];
                }
            }
            return auxHudVisibile;
        }
        
        return false;
    };
    
    /*
    function:setCurrentSelectedElement - set the current selected node in Live View. This is the live view 
                                         element corresponding to the selected element in DW DOM.

    Arguments: curElement - DOM element which is to be set as the current selection

    Return: none
    */
    LiveViewObjectForExtensions.prototype.setCurrentSelectedElement = function (curElement, elementIsNonEditable) {
        this.currentSelectedElement = curElement;
        if (elementIsNonEditable === 'true') {
            this.isANonEditableElement = true;
        } else {
            //we need to check whether the element is in editable region in case of template instance
            this.isANonEditableElement = window.liveEditTemplateInstance ? !this.isElementInsideEditableRegion(curElement) : false;
        }
    };

    /*
    function:isLiveEditInProgress - returns if Live Edit is in progress.

    Arguments: none

    Return: Returns true if Live Edit is in progress, false otherwise.
    */
    LiveViewObjectForExtensions.prototype.isLiveEditInProgress = function () {
        return this.liveEditInProgress;
    };

    /*
    function:setLiveEditInProgress - set if Live Edit is in progress.

    Arguments: inProgress - boolean. true if Live Edit is in progress, false otherwise.

    Return: none
    */
    LiveViewObjectForExtensions.prototype.setLiveEditInProgress = function (inProgress) {

        if(this.liveEditInProgress === inProgress)
            return;

        this.liveEditInProgress = inProgress;
        window.DWSetLiveEditingInProgress(this.liveEditInProgress, 0 /*dummy*/, 0 /*dummy*/);
    };

    /*
    function:setLiveEditCommitPending - set if Live Edit Commit Pending.
    Arguments: isPending - boolean. true if Live Edit commit is Pending, false otherwise.
    Return: none
    */
   LiveViewObjectForExtensions.prototype.setLiveEditCommitPending = function (isPending) {

        if(this.liveEditCommitPending === isPending)
            return;

        this.liveEditCommitPending = isPending;
        window.DWSetLiveEditCommitPending(this.liveEditCommitPending);
};

    /*
    function:setSelectionInTemplateInstance - Function to set the selection in a Template Instance Page. Sets the passed element as
                                                our selected element and sets whether it is editable based on whether the element is
                                                is inside Editable Region or not.
    Arguments: curElement - DOM element which is to be set as the current selection

    Return: none
    */
    LiveViewObjectForExtensions.prototype.setSelectionInTemplateInstance = function (curElement) {
        var isNonEditable = !this.isElementInsideEditableRegion(curElement);
        if (isNonEditable)
            this.setCurrentSelectedElement(curElement, 'true');
        else
            this.setCurrentSelectedElement(curElement, 'false');
    };

    /*
    function:isElementEditable - Check whether the passed element is editable from live view.
                                 check is based on the presence of dw node ID

    Arguments: curElement - DOM element

    Return: true if editable, false otherwise
    */
    LiveViewObjectForExtensions.prototype.isElementEditable = function (element) {
        var editable = false;
        if (element && !this.isANonEditableElement) {
            var tagUniqId = element.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
            if (tagUniqId && tagUniqId.length > 0) {
                editable = true;
            }
        }

        return editable;
    };

    /*
    function:isAnyChildInsideEditableRegion - Checks if the any child of passed element is inside DW Template 
                                              Editable Region bounds

    Arguments: element

    Return: true if the any child element is inside an Editable region, false otherwise
    */
    LiveViewObjectForExtensions.prototype.isAnyChildInsideEditableRegion = function (element) {
        if (!element)
            return false;

        var childNodes = element.childNodes;
        if (childNodes) {
            for (var i = 0; i < childNodes.length; i++) {
                var child = childNodes[i];
                if (child && (child.nodeType === Node.COMMENT_NODE) && (child.nodeValue.indexOf(DW_LIVEEDIT_CONSTANTS.EditableRegionBeginText) === 0)) {
                    return true;
                }
            }
        }
        return false;
    }
    
    /*
    function:isElementInsideEditableRegion - Checks if the passed element is inside DW Template 
                                             Editable Region bounds

    Arguments: element

    Return: true if the element is inside an Editable region, false otherwise
    */
    LiveViewObjectForExtensions.prototype.isElementInsideEditableRegion = function (element) {
        if (!element) {
            return false;
        }

        var isInsideEditableRegion = false;
        var curNode = element;

        if (this.isAnyChildInsideEditableRegion(element))
            return true;

        while (curNode) {
            //DW templates uses comments for marking regions
            if (curNode.nodeType === Node.COMMENT_NODE) {
                // if we reach ER End first, then element is outside of ER.
                if (curNode.nodeValue.indexOf(DW_LIVEEDIT_CONSTANTS.EditableRegionEndText) === 0) {
                    break;
                }

                // if we reach ER Begin first, then element is inside ER.
                if (curNode.nodeValue.indexOf(DW_LIVEEDIT_CONSTANTS.EditableRegionBeginText) === 0) {
                    isInsideEditableRegion = true;
                    break;
                }
            }

            // If no previous sibling, then take the parent and search
            if (curNode.previousSibling) {
                curNode = curNode.previousSibling;
            } else {
                curNode = curNode.parentNode;
            }
        }
        
        return isInsideEditableRegion;
    };

    // Get Live view scroll bar width
	LiveViewObjectForExtensions.prototype.getScrollBarWidth = function () {
		if (this.scrollBarWidth) {
			return this.scrollBarWidth;
		}

		// Add dummy container and bigger inner element to
		// calculate scroll bar width.
		var inner = document.createElement('p');
		inner.style.width = "100%";
		inner.style.height = "200px";
        inner.style.padding = "0px";
        inner.style.margin = "0px";
        inner.style.border = "0px";

		var outer = document.createElement('div');
		outer.style.position = "absolute";
        outer.style.padding = "0px";
        outer.style.margin = "0px";
        outer.style.border = "0px";
		outer.style.top = "0px";
		outer.style.left = "0px";
		outer.style.visibility = "hidden";
		outer.style.width = "200px";
		outer.style.height = "150px";
		outer.style.overflow = "hidden";

		outer.appendChild(inner);
		document.body.appendChild(outer);

		// Calculate visible width of elements
		var w1 = inner.offsetWidth;
		outer.style.overflow = 'scroll';
		var w2 = inner.offsetWidth;
		if (w1 === w2) {
			w2 = outer.clientWidth;
		}

		// Remove dummy elements
		document.body.removeChild(outer);

		// Remember scroll bar width
		this.scrollBarWidth = (w1 - w2);

		return this.scrollBarWidth;
	};

	// Get Scroll bar state
	LiveViewObjectForExtensions.prototype.getScrollBarState = function () {
		var result = {vScrollbar: true, hScrollbar: true, hScrollbarHeight: 0, vScrollbarWidth: 0};
		try {
			var root = document.compatMode === 'BackCompat' ? document.body : document.documentElement;
			result.vScrollbar = root.scrollHeight > root.clientHeight;
			result.hScrollbar = root.scrollWidth > root.clientWidth;
			result.hScrollbarHeight = this.getScrollBarWidth();
			result.vScrollbarWidth = this.getScrollBarWidth();
		} catch (e) {}
		return (result);
	};
    
    // Get view port size
    // Taken from InspectModeUtils and exposing as a generic function.
    LiveViewObjectForExtensions.prototype.getViewPortSize = function () {
        
        var scrollState = this.getScrollBarState(),
            viewPortSize = { height: window.innerHeight, width: window.innerWidth };

        if (scrollState.hScrollbar) {
            viewPortSize.height = (window.innerHeight - scrollState.hScrollbarHeight);
        }

        if (scrollState.vScrollbar) {
            viewPortSize.width = (window.innerWidth - scrollState.vScrollbarWidth);
        }

        return viewPortSize;
    };
    
    /*
    function:getElementRect - Returns the containing rect for the element in the document

    Arguments: elem - DOM element

    Return: containing rect(top, left, width, height) of the element
    */
    LiveViewObjectForExtensions.prototype.getElementRect = function (elem) {
        var rect = { top : 0, left : 0, width : 0, height : 0 };
        if (elem) {
            var boudingRect = elem.getBoundingClientRect();
            rect.top = boudingRect.top + window.scrollY;
            rect.left = boudingRect.left + window.scrollX;
            rect.width = boudingRect.right - boudingRect.left;
            rect.height = boudingRect.bottom - boudingRect.top;
        }
        return rect;
    };
    
    /*
    function:clearBrowserUndoRedos - Clears browser Undo/Redo Stack. Undo/Redo goes to Titandoc only when 
                                     browser undo/redo stack is empty.
    Arguments: none

    Return: none
    */
    LiveViewObjectForExtensions.prototype.clearBrowserUndoRedos = function () {
        //clear the stack with the new command
        document.execCommand("ClearUndoRedoOperations", false, null);

        //DW updates the Live View Undo/Redo status on selection change
        //so initiate a selection change
        var selectionChangeEvent = new CustomEvent("selectionchange");
        document.dispatchEvent(selectionChangeEvent);
    };
    /*
    function:isElementHidden -  check if the element is hidden
    Arguments: node

    Return: return true if element is hidden
    */
    LiveViewObjectForExtensions.prototype.isElementHidden = function (node) {
        var elemRect = this.getElementRect(node);
        if (elemRect && (elemRect.height === 0 || elemRect.width === 0)) {
            return true;
        }
        var style = window.getComputedStyle(node);
        if (!style) {
            return true;
        }
        return (style.display.toLowerCase() === 'none');
    };
    
    /*
    function:setSelectionOnElement - Set selection on an element

    Arguments: node - element to be set selection
               dontClear - if true, exisitng selection will not be cleared

    Return: whether we are able to set selection or not.
    */
    LiveViewObjectForExtensions.prototype.setSelectionOnElement = function (node, dontClear) {
        // if this list grows will maintain it in constants. 
        if (!node || node.nodeType !== Node.ELEMENT_NODE) {
            return false;
        }
        var nodeTagName = node.tagName;
        if (nodeTagName !== "SCRIPT" && nodeTagName !== "BR" && nodeTagName.toLowerCase() !== DW_LIVEEDIT_CONSTANTS.GlobalContainerDiv && !this.isElementHidden(node)) {
            //first clear any text selection if present
            if (!dontClear && window.getSelection()) {
                window.getSelection().removeAllRanges();
            }
            
            var dwId = node.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
            if (dwId) {
                this.dwObject.DWSMSetSelectionOnNodeWithId(dwId, true);
            } else {
                this.setCurrentSelectedElement(node, 'true');
                //send a selection changed event to notify extensions
                var dwSelectionEvent = {};
                dwSelectionEvent.type = DW_EVENTS.SelectionChanged;
                this.messageToExtensions(dwSelectionEvent);
            }
            return true;
        }
        return false;
    };

    LiveViewObjectForExtensions.prototype.callShowHud = function () {
        var dwSelectionEvent = {};
        dwSelectionEvent.type = DW_EVENTS.ShowHud;
        this.messageToExtensions(dwSelectionEvent);
    };

    /*
    function:showExtensionById - display the extension

    Arguments: extensionID - id of the extension

    Return: none
    */
    LiveViewObjectForExtensions.prototype.showExtensionById = function (extensionID) {
        var extensionIndex = this.getExtensionIndexById(extensionID);
        if (extensionIndex >= 0) {
            //show only if enabled
            if (this.liveViewExtensionsArray[extensionIndex].enabled === true) {
                var extensionIFrame = this.liveViewExtensionsArray[extensionIndex].frame;
                if (extensionIFrame && extensionIFrame.style) {
                    extensionIFrame.style.display = 'block';
                }
            }
        }
    };

    /*
    function:hideExtensionById - hide the extension

    Arguments: extensionID - id of the extension

    Return: none
    */
    LiveViewObjectForExtensions.prototype.hideExtensionById = function (extensionID) {
        var extensionIFrame = this.getExtensionIFrameById(extensionID);
        if (extensionIFrame && extensionIFrame.style) {
            extensionIFrame.style.display = 'none';
        }
    };
    
    /*
    function:isExtensionVisible - return the visibility of extension by ID

    Arguments: extensionID - id of the extension

    Return: true if visible, false otherwise
    */
    LiveViewObjectForExtensions.prototype.isExtensionVisible = function (extensionID) {
        var isVisible = false;
        var extensionIFrame = this.getExtensionIFrameById(extensionID);
        if (extensionIFrame && extensionIFrame.style) {
            isVisible = (extensionIFrame.style.display !== 'none');
        }
        
        return isVisible;
    };
    
    /*
    function:unloadExtensionById - Unload a given extension

    Arguments: none

    Return: none
    */
    LiveViewObjectForExtensions.prototype.unloadExtensionById = function (extensionID) {
        var i = this.getExtensionIndexById(extensionID);
        var extensionIFrame = this.getExtensionIFrameById(extensionID);
        if (extensionIFrame) {
            extensionIFrame.parentNode.removeChild(extensionIFrame);
            if (i >= 0) {
                this.liveViewExtensionsArray.splice(i, 1);
            }
        }
    };

    /*
    function:disableAllExtensions - disable all extensions

    Arguments: none

    Return: none
    */
    LiveViewObjectForExtensions.prototype.disableAllExtensions = function (params) {
        if (params === DW_EVENTS_PARAM.CodeViewEdit || params === DW_EVENTS_PARAM.DragEnter) {
            // If in code view, don't disable responsive exension
            this.disableResponsiveExt = false;
        } else {
            this.disableResponsiveExt = true;
        }
        var i;
        for (i = 0; i < this.liveViewExtensionsArray.length; i++) {
            if (this.liveViewExtensionsArray[i] && this.isDisableAllowedOnExtension(this.liveViewExtensionsArray[i].id, params)) {
                var extensionIFrame = this.liveViewExtensionsArray[i].frame;
                this.liveViewExtensionsArray[i].enabled = false;
                if (extensionIFrame && extensionIFrame.style) {
                    extensionIFrame.style.display = 'none';
                }
            }
        }
    };
    
    /*
    function:isDisableAllowedOnExtension - check if extension can be disabled
    
    Arguements: extension id
    
    Return: True/False
    */
    LiveViewObjectForExtensions.prototype.isDisableAllowedOnExtension = function (id, params) {
        if (id === RESPONSIVE_FRAMEWORK.ExtensionId && (params === DW_EVENTS_PARAM.CodeViewEdit || params === DW_EVENTS_PARAM.DragEnter)) {
            return false;
        }
        return true;
    };

    /*
    function:enableAllExtensions - enable all extensions

    Arguments: none

    Return: none
    */
    LiveViewObjectForExtensions.prototype.enableAllExtensions = function () {
        this.disableResponsiveExt = false;
        var i;
        for (i = 0; i < this.liveViewExtensionsArray.length; i++) {
            if (this.liveViewExtensionsArray[i]) {
                this.liveViewExtensionsArray[i].enabled = true;
            }
        }
    };

    /*
    function:positionExtensionById - position an extension in the document

    Arguments: extension ID, top, left, width and height to be set

    Return: none
    */
    LiveViewObjectForExtensions.prototype.positionExtensionById = function (extensionID, top, left, width, height) {
        var extensionIndex = this.getExtensionIndexById(extensionID);
        if (extensionIndex >= 0) {
            //position only if enabled
            if (this.liveViewExtensionsArray[extensionIndex].enabled === true) {
                var extensionIFrame = this.liveViewExtensionsArray[extensionIndex].frame;
                //if document body or html element is positioned relative
                //then their top left may not be (0,0). We should reduce this
                //in our computations
                var topOffset = 0, leftOffset = 0, documentHtmlStyle = null;
                var documentBodyStyle = window.getComputedStyle(document.body);
                var documentHTMLElement = document.getElementsByTagName("html")[0];
                if (documentHTMLElement) {
                    documentHtmlStyle = window.getComputedStyle(documentHTMLElement);
                }
                if (documentBodyStyle.position === "relative") {
                    var bodyRect = this.getElementRect(document.body);
                    topOffset = bodyRect.top;
                    leftOffset = bodyRect.left;
                } else if (documentHtmlStyle && documentHtmlStyle.position === "relative") {
                    var htmlRect = this.getElementRect(documentHTMLElement);
                    topOffset = htmlRect.top;
                    leftOffset = htmlRect.left;
                }
                
                if (extensionIFrame && extensionIFrame.style) {
                    extensionIFrame.style.top = (top - topOffset) + "px";
                    extensionIFrame.style.left = (left - leftOffset) + "px";
                    extensionIFrame.style.height = height + "px";
                    extensionIFrame.style.width = width + "px";
                    extensionIFrame.style.display = "block";
                }
            }
        }
    };

    /*
    function:getExtensionIFrameById - get the iframe for the extension in the document

    Arguments: extension ID

    Return: IFrame element for the extention if found
    */
    LiveViewObjectForExtensions.prototype.getExtensionIFrameById = function (extensionID) {
        var extensionIFrame = null;
        var extensionIndex = this.getExtensionIndexById(extensionID);
        if (extensionIndex >= 0) {
            extensionIFrame = this.liveViewExtensionsArray[extensionIndex].frame;
        }
        return extensionIFrame;
    };

    /*
    function:getExtensionIndexById - get the index for the extension in the list

    Arguments: extension ID

    Return: index of the extension if found, -1 otherwise
    */
    LiveViewObjectForExtensions.prototype.getExtensionIndexById = function (extensionID) {
        var extensionIndex = -1;
        var i;
        if (extensionID && extensionID.length > 0) {
            for (i = 0; i < this.liveViewExtensionsArray.length; i++) {
                if (this.liveViewExtensionsArray[i] && this.liveViewExtensionsArray[i].id === extensionID) {
                    extensionIndex = i;
                    break;
                }
            }
        }
        return extensionIndex;
    };

    /*
    function:fetchExtensionIFrameCSS - return the CSS to be applied to each IFrame for an Extension

    Arguments: none

    Return: string
    */
    LiveViewObjectForExtensions.prototype.fetchExtensionIFrameCSS = function () {
        return "animation:none;animation-delay:0;animation-direction:normal;animation-duration:0;animation-fill-mode:none;animation-iteration-count:1;animation-name:none;animation-play-state:paused;animation-timing-function:ease;backface-visibility:hidden;background:0;background-attachment:scroll;background-clip:border-box;background-color:transparent;background-image:none;background-origin:padding-box;background-position:00;background-position-x:0;background-position-y:0;background-repeat:repeat;background-size:0;border:0;border-style:none;border-width:medium;border-color:transparent;border-bottom:0;border-bottom-color:transparent;border-bottom-left-radius:0;border-bottom-right-radius:0;border-bottom-style:none;border-bottom-width:medium;border-collapse:separate;border-image:none;border-left:0;border-left-color:transparent;border-left-style:none;border-left-width:medium;border-radius:0;border-right:0;border-right-color:transparent;border-right-style:none;border-right-width:medium;border-spacing:0;border-top:0;border-top-color:transparent;border-top-left-radius:0;border-top-right-radius:0;border-top-style:none;border-top-width:medium;bottom:0;box-shadow:none;box-sizing:content-box;caption-side:top;clear:none;clip:auto;color:#000000;columns:auto;column-count:auto;column-fill:balance;column-gap:normal;column-rule:mediumnonecurrentColor;column-rule-color:currentColor;column-rule-style:none;column-rule-width:none;column-span:1;column-width:auto;content:normal;counter-increment:none;counter-reset:none;cursor:auto;direction:ltr;display:none;empty-cells:show;float:none;font:normal;font-family:tahoma;font-size:medium;font-style:normal;font-variant:normal;font-weight:normal;height:auto;hyphens:none;left:0px;letter-spacing:normal;line-height:normal;list-style:none;list-style-image:none;list-style-position:outside;list-style-type:disc;margin:0;margin-bottom:0;margin-left:0;margin-right:0;margin-top:0;max-height:none;max-width:none;min-height:0;min-width:0;opacity:1;orphans:0;outline:0;outline-color:auto;outline-style:auto;outline-width:auto;overflow:visible;overflow-x:visible;overflow-y:visible;padding:0;padding-bottom:0;padding-left:0;padding-right:0;padding-top:0;page-break-after:auto;page-break-before:auto;page-break-inside:auto;perspective:none;perspective-origin:50%50%;position:absolute;right:auto;tab-size:8;table-layout:auto;text-align:left;/*CEF1650 asserts, disabling ->*//*text-align-last:auto;*/text-decoration:none;text-decoration-color:transparent;text-decoration-line:none;text-decoration-style:solid;text-indent:0;text-shadow:none;text-transform:none;top:0px;transform:none;transform-style:flat;transition:none;transition-delay:0s;transition-duration:0s;transition-property:none;transition-timing-function:ease;unicode-bidi:normal;vertical-align:baseline;visibility:visible;white-space:normal;widows:0;width:auto;word-spacing:normal;z-index:99999999999999999999999999;user-select:none";
    };

    /*
    function:getExtensionContainer - Get the DOM Element which contains all the Extensions

    Arguments: none

    Return: DOMElement
    */
    LiveViewObjectForExtensions.prototype.getExtensionContainer = function () {
        var extensionContainer = document.getElementsByTagName(DW_LIVEEDIT_CONSTANTS.GlobalContainerDiv)[0];
        if (!extensionContainer) {
            extensionContainer = document.createElement(DW_LIVEEDIT_CONSTANTS.GlobalContainerDiv);
            //set position property to default value to override * selector
            extensionContainer.style.cssText = "position:static;";
        }
        return extensionContainer;
    };

    /*
    function:configureExtensionResourcePath - configure the extensionsResourcePath

    Arguments: none

    Return: none
    */
    LiveViewObjectForExtensions.prototype.configureExtensionResourcePath = function () {
        if (window.location && window.location.protocol !== "file:") {
            //If it is not a file: URL, browser will not load file: URLs in iFrames due to cross origin policy.
            //So our extensions will not load in this case. Here we are faking that our extensions
            //are present in the remote host. We will provide these files from DW CEF resource loader
            ///when we get a load request. SO we need to identify our extension files in the URL.
            //Adding a placeholder here to identify our extension resources. DW CEF resource loader 
            //will look for this placeholder and replace as needed if found. And the browser will think
            //that this is coming from remote host, thus bypassing cross origin policy
            this.liveExtensionsConfigPath = window.location.protocol + "//" + window.location.hostname + "//";
            this.liveExtensionsConfigPath += DW_LIVEEDIT_CONSTANTS.ExtensionsResourcePathPlaceHolder;
        } else {
            this.liveExtensionsConfigPath = window.extensionsResourcePath;
        }
        
    };
    
    LiveViewObjectForExtensions.prototype.attachResponsiveLayoutBridgingMethods = function () {
        /*
            function: getAllInfoForResponsiveInsert - required while Inserting 
            Arguments: none
            Returns: All the class names to be applied, read from the xml file.
        */
        window.getAllInfoForResponsiveInsert = function () {
            var retObj = {};
            if (window.liveViewExtensionsObject) {
                var extensionIFrame = window.liveViewExtensionsObject.getExtensionIFrameById(RESPONSIVE_FRAMEWORK.ExtensionId);
                if (extensionIFrame) {
                    var wind = extensionIFrame.contentWindow;
                    if (wind && wind.responsiveLayoutManipulator) {
                        retObj = wind.responsiveLayoutManipulator.getAllInfoForResponsiveInsert();
                    }
                }
            }
            return retObj;
        };
        
        window.canElementBeHidden = function () {
            var canHideElement = false;
            if (window.liveViewExtensionsObject) {
                var extensionIFrame = window.liveViewExtensionsObject.getExtensionIFrameById(RESPONSIVE_FRAMEWORK.ExtensionId);
                if (extensionIFrame) {
                    var wind = extensionIFrame.contentWindow;
                    if (wind && wind.responsiveLayoutManipulator) {
                        canHideElement = wind.responsiveLayoutManipulator.canHideSelectedElement();
                    }
                }
            }
            return canHideElement;
        };
        
        window.isElementHidden = function () {
            var isElementHidden = false;
            if (window.liveViewExtensionsObject) {
                var extensionIFrame = window.liveViewExtensionsObject.getExtensionIFrameById(RESPONSIVE_FRAMEWORK.ExtensionId);
                if (extensionIFrame) {
                    var wind = extensionIFrame.contentWindow;
                    if (wind && wind.responsiveLayoutManipulator) {
                        isElementHidden = wind.responsiveLayoutManipulator.isElementHidden();
                    }
                }
            }
            return isElementHidden;
        };
		
		/*
            function: getGridSizesWithPrefix - For adding menu items to Scrubber context menu and Window Size menu for Bootstrap pages
            Arguments: none
            Returns: Bootstrap breakpoints value with the size prefix as a string eg. xs-0-767;sm-768-991
        */
		window.getGridSizesWithPrefix = function () {
			var retObj = "";
            if (window.liveViewExtensionsObject) {
                var extensionIFrame = window.liveViewExtensionsObject.getExtensionIFrameById(RESPONSIVE_FRAMEWORK.ExtensionId);
                if (extensionIFrame) {
                    var wind = extensionIFrame.contentWindow;
                    if (wind && wind.responsiveLayoutManipulator) {
                        retObj = wind.responsiveLayoutManipulator.getGridSizesWithPrefix();
                    }
                }
            }
            return retObj;
        };
    };

    /*
    function:loadExtensionById - find the extensionDetails and load the extension

    Arguments: extensionID

    Return: none
    */
    LiveViewObjectForExtensions.prototype.loadExtensionById = function (extensionID, loadExtensionAtBack) {
        //make sure we have the details of extensions preloaded
        if (!window.extensionsDetail || !window.extensionsResourcePath) {
            return;
        }
        
        var extensionsDetails = window.extensionsDetail,
            i = 0;
        
        for (i = 0; i < extensionsDetails.length; i++) {
            if (extensionID === extensionsDetails[i].id) {
                this.loadExtension(extensionsDetails[i], loadExtensionAtBack);
                break;
            }
        }
        if (extensionID === RESPONSIVE_FRAMEWORK.ExtensionId) {
            this.attachResponsiveLayoutBridgingMethods();
        }
    };
    
    /*
    function:loadExtension - load an extension to an iframe and add it to document

    Arguments: extensionDetails

    Return: none
    */
    LiveViewObjectForExtensions.prototype.loadExtension = function (extensionDetails, loadExtensionAtBack) {
        //create an iframe for each extension and set its source
        var src = extensionDetails.src,
            extensionID = extensionDetails.id,
            extensionType = extensionDetails.type;
        
        if (this.getExtensionIndexById(extensionID) !== -1) {
            // don't try to load an already loaded extension
            return;
        }

        var extensionIFrame = document.createElement("iframe"),
            extensionContainer = this.getExtensionContainer();

        extensionIFrame.src = this.liveExtensionsConfigPath + src;
        extensionIFrame.style.cssText = this.fetchExtensionIFrameCSS();
        extensionIFrame.id = extensionID;
        //add identifying attribute for extension iframes
        extensionIFrame.setAttribute("dwExtensionFrame", "true");
        extensionIFrame.setAttribute("random", Math.random());
        if (extensionType === "auxiliary") {
            var tags =  extensionDetails.tags;
            if (tags) {
                if (tags.indexOf(" ") !== -1) {
                    tags = tags.split(" ");
                }
                if (typeof tags === "string") {
                    this.auxiliaryIframeExists[tags] = true;
                } else if (typeof tags === "object") {
                    var j;
                    for (j = 0; j < tags.length; j++) {
                        var currentTag = tags[j].toLowerCase();
                        this.auxiliaryIframeExists[currentTag] = true;
                    }
                }
            }
        }
        //add the iframe inside the container
        if (typeof loadExtensionAtBack !== "undefined" && loadExtensionAtBack === true) {
            if (extensionContainer.firstChild) {
                extensionContainer.insertBefore(extensionIFrame, extensionContainer.firstChild);
            } else {
                extensionContainer.appendChild(extensionIFrame);
            }
        } else {
            extensionContainer.appendChild(extensionIFrame);
        }
        extensionIFrame = document.getElementById(extensionID);
        this.liveViewExtensionsArray.push({id: extensionID, frame: extensionIFrame, enabled: true});

        //attach dwObject to the extension iframe
        if (extensionIFrame.contentWindow) {
            extensionIFrame.contentWindow.liveViewObject = this;
            extensionIFrame.contentWindow.dwObject = this.dwObject;
            extensionIFrame.contentWindow.liveExtensionsConfigPath = this.liveExtensionsConfigPath;
            extensionIFrame.contentWindow.isExtensionInitialized = true;
            //If it's the FluidGrid Extension and the current document is an FG document, attach the FG data required to the window 
            if (extensionIFrame.id === "FluidGrid" && window.liveEditFGInstance) {
                try {
                    extensionIFrame.contentWindow.gFluidGridData = gFluidGridData;
                    extensionIFrame.contentWindow.gFluidGridUrl = gFluidGridUrl;
                } catch (e) {
                    window.writeDWLog("Exception in recognizing FluidGrid variables " + e);
                }
            }
        }
    };
    
    /*
    function:loadRequiredExtensions - load each extension to an iframe and add it to document

    Arguments: none

    Return: none
    */
    LiveViewObjectForExtensions.prototype.loadRequiredExtensions = function () {
        var i;
      
        //make sure we have the details of extensions preloaded
        if (!window.extensionsDetail || !window.extensionsResourcePath) {
            return;
        }

        var extensionsDetails = window.extensionsDetail,
            extensionID = "";

        for (i = 0; i < extensionsDetails.length; i++) {
            extensionID = extensionsDetails[i].id;

            if (extensionID === "FluidGrid" && !window.liveEditFGInstance) {
                // Don't load the Fluid Grid Extension for NonFG pages
                continue;
            }
			
			if (extensionID !== RESPONSIVE_FRAMEWORK.ExtensionId) {
				this.loadExtension(extensionsDetails[i]);
			} else if (window.supportedResponsiveFrameworkInfo) {
				// Don't load the Responsive Layout Extension for pages that don't have a framework we support
                // this is expensive so let all the other extensions load.
                this.scheduleLoadingOfResponsiveFrameworkInfo();
			}
        }
    };
    /*
    function:didStyleSheetsChange - Figure out if there is a change in the window stylesheets from when called previously
    Arguments: none
    Return: boolean: true if changed
    */
    LiveViewObjectForExtensions.prototype.didStyleSheetsChange = function () {
        var differentStyleSheetList = false,
            newHrefList = [],
            idx = 0;
        // try the various href
        for (idx = 0; idx < window.document.styleSheets.length; ++idx) {
            var styleSheet = window.document.styleSheets[idx];
            var fileName = styleSheet.href;
            if (!fileName) {
                if (styleSheet.cssRules
                        && styleSheet.cssRules.length === 1
                        && styleSheet.cssRules[0].type === CSSRule.IMPORT_RULE
                        && styleSheet.cssRules[0].href) {
                    fileName = styleSheet.cssRules[0].href;
                }
            }
            if (fileName) {
                newHrefList.push(fileName);
            }
        }
        if (window.liveViewExtensionsObject.styleSheetCount !== window.document.styleSheets.length
                || window.liveViewExtensionsObject.styleSheethref.length !== newHrefList.length) {
            differentStyleSheetList = true;
        }
        if (!differentStyleSheetList) {
            // try to match each href to find a difference
            for (idx = 0; idx < newHrefList.length; ++idx) {
                if (window.liveViewExtensionsObject.styleSheethref.indexOf(newHrefList[idx]) === -1) {
                    differentStyleSheetList = true;
                    break;
                }
            }
        }

        // update the stored information
        window.liveViewExtensionsObject.styleSheethref = newHrefList;
        window.liveViewExtensionsObject.styleSheetCount = window.document.styleSheets.length;
        return differentStyleSheetList;
    };
    /*
    function:scheduleLoadingOfResponsiveFrameworkInfo - load each of the Responsive Framework XMLs and then proceed to identify the document

    Arguments: none

    Return: none
    */
    LiveViewObjectForExtensions.prototype.scheduleLoadingOfResponsiveFrameworkInfo = function () {
        var self = this;

        if (this.responsiveLoadingTimeout) {
            clearTimeout(this.responsiveLoadingTimeout);
        }
		
		var dwExtension = this.dwObject;
        
        this.responsiveLoadingTimeout = setTimeout(function () {
            var callback = function shouldReloadResponsiveFramework(shouldReload){
                var callbackFunction = function loadResponsiveFramework(isCurrentDocumentTemplate) {                                                                       if (!self.disableResponsiveExt || isCurrentDocumentTemplate) {
                    var shouldLookupFramework = shouldReload || self.didStyleSheetsChange();
                    if (shouldReload && dwExtension) {
                        dwExtension.setReloadBootstrapFramework(false);
                    }
                    if (shouldLookupFramework) {
                        ResponsiveFrameworkLookup.frameworkXML = [];
                        var lookForXMls = window.supportedResponsiveFrameworkInfo.length,
                            i = 0;
                        if (self.responsiveFrameworkXMLs.length === 0) {
                            var parser = new DOMParser();
                            for (i = 0; i < lookForXMls; i++) {
                                var xmlDoc = parser.parseFromString(window.supportedResponsiveFrameworkInfo[i].fileStr, "text/xml");               self.responsiveFrameworkXMLs.push(xmlDoc);
                            }
                        }
                        ResponsiveFrameworkLookup.frameworkXML = self.responsiveFrameworkXMLs;
                        ResponsiveFrameworkLookup.lookForResponsiveFrameworkCSS();
                    }
                }
            }
                dwExtension.isTemplate(callbackFunction);
            };
            if (dwExtension) {
                dwExtension.shouldReloadBootstrapFramework(callback);
            } else {
                callback(false);
            }
			
            self.responsiveLoadingTimeout = null;
        }, 500);
    };

    LiveViewObjectForExtensions.prototype.isBootstrapResponsiveFrameworkLoaded = function () {
        return ResponsiveFrameworkLookup.isBootstrapResponsiveFrameworkLoaded();
    };

    LiveViewObjectForExtensions.prototype.getCompatibleBootstrapVersion = function () {
        return window.compatibleBootstrapVersion;
    };

    LiveViewObjectForExtensions.prototype.getRWDSubCategoryStr = function () {
        return DW_ICE_HEADLIGHTS.RWD_BOOTSTRAP + window.bootstrapVersion;
    };
    /*
    function:messageToExtensions - send a message to all extensions

    Arguments: messageDetails: type(string), params(string)

    Return: none
    */

    LiveViewObjectForExtensions.prototype.messageToExtensions = function (messageDetails) {
        var i;
        messageDetails.dwExtensionEvent = true;
        for (i = 0; i < this.liveViewExtensionsArray.length; i++) {
            if (this.liveViewExtensionsArray[i] && this.liveViewExtensionsArray[i].enabled === true) {
                if (this.liveViewExtensionsArray[i].frame && this.liveViewExtensionsArray[i].frame.contentWindow) {
                    this.liveViewExtensionsArray[i].frame.contentWindow.postMessage(messageDetails, "*");
                }
            }
        }
    };
    
    /*
    function:handleEventForParent - Extension event to Parent. Needs to be handled by the parent document

    Arguments: messageDetails: type(string), params(string)

    Return: none
    */
    LiveViewObjectForExtensions.prototype.handleEventForParent = function (messageDetails) {
        if (messageDetails.type === DW_PARENT_EVENT.UNLOAD_EXTENSION_BY_ID) {
            this.unloadExtensionById(messageDetails.extensionID);
        } else if (messageDetails.type === DW_PARENT_EVENT.RESPONSIVE_HIDDEN_ELEMENT_COUNT) {
            window.dwHiddenElementExist = messageDetails.hiddenElementExist;
        }
    };
	
	/*
    function:reInitializeLiveViewExtensions - send a message to all extensions

    Arguments: reInitialize extensions if needed

    Return: none
    */

    LiveViewObjectForExtensions.prototype.reInitializeLiveViewExtensions = function () {
        var i;
		for (i = 0; i < this.liveViewExtensionsArray.length; i++) {
            if (this.liveViewExtensionsArray[i]) {
				var extensionIFrame = this.liveViewExtensionsArray[i].frame;
                if (extensionIFrame && extensionIFrame.contentWindow) {
                    if (!extensionIFrame.contentWindow.isExtensionInitialized) {
						extensionIFrame.contentWindow.liveViewObject = this;
						extensionIFrame.contentWindow.dwObject = this.dwObject;
						extensionIFrame.contentWindow.liveExtensionsConfigPath = this.liveExtensionsConfigPath;
                        if (extensionIFrame && extensionIFrame.style) {
                            extensionIFrame.style.display = 'none';
                        }
						extensionIFrame.contentWindow.isExtensionInitialized = true;
						//If it's the FluidGrid Extension ans the current document is an FG document, attach the FG data required to the window 
						if (extensionIFrame.id === "FluidGrid" && window.liveEditFGInstance) {
							try {
								extensionIFrame.contentWindow.gFluidGridData = gFluidGridData;
								extensionIFrame.contentWindow.gFluidGridUrl = gFluidGridUrl;
							} catch (e) {
								this.logDebugMsg("Exception in recognizing FluidGrid variables " + e);
							}
						}
					}
                }
            }
        }
    };

	/*
    function:deInitializeLiveViewExtensions: deInitialize extensions

    Arguments: none

    Return: none
    */

    LiveViewObjectForExtensions.prototype.deInitializeLiveViewExtensions = function () {

        for (var i = 0; i < this.liveViewExtensionsArray.length; ++i) {
            if (this.liveViewExtensionsArray[i]) {
                var extensionIFrame = this.liveViewExtensionsArray[i].frame;
                var wnd = extensionIFrame ? extensionIFrame.contentWindow : null;
                if(!wnd || !wnd.deinitialize)
                    continue;

                wnd.deinitialize();
            }
        }
    };

    /*
    function:initializeLiveViewExtensionObjects - function to initialize the objects exposed by our infrastructure
                                                  to the passed extension frame
    Arguments: id of extension

    Return: none
    */

    LiveViewObjectForExtensions.prototype.initializeLiveViewExtensionObjects = function (extensionID) {
        var extensionIFrame = this.getExtensionIFrameById(extensionID);
        if (extensionIFrame && extensionIFrame.contentWindow) {
            extensionIFrame.contentWindow.liveViewObject = this;
            extensionIFrame.contentWindow.dwObject = this.dwObject;
            extensionIFrame.contentWindow.liveExtensionsConfigPath = this.liveExtensionsConfigPath;
            extensionIFrame.contentWindow.isExtensionInitialized = true;
        }
    };

    LiveViewObjectForExtensions.prototype.initializeResponsiveFrameworkConstants = function () {
        // For responsive layout extension, which is loaded with a timeout delay set, it need to know whether loading of all extensions
        // has been disabled. If so, then don't load the extensions.
        this.disableResponsiveExt = false;
        this.styleSheetCount = -1;
        this.styleSheethref = [];
        this.responsiveLoadingTimeout = null;
        this.responsiveFrameworkXMLs = [];
    };
    
    LiveViewObjectForExtensions.prototype.setLiveViewResizing = function (resizing) {
        this.liveViewResizing = resizing;
    };
    
    LiveViewObjectForExtensions.prototype.getLiveViewResizing = function (resizing) {
        return this.liveViewResizing;
    };
    /*
    function:initLiveViewExtensionsInfrastructure - global function to set up the infrastructure and load
                                                    the extensions
    Arguments: none

    Return: none
    */
    var initLiveViewExtensionsInfrastructure = function () {
        /* Utility functions */

        /*
        function:updateSelectedElement - Find the element base don arguments 
                                         and set it as current selection for extensions.    
        Arguments: uniqNodeId - unique node id of the element
                   selectorString - if the node id is not present, then use selector
        Return: none
        */
        var updateSelectedElement =  function (uniqNodeId, selectorString, elementIsNonEditable) {
            var targetElement = null;
            var querySelectorString = "";
            if (uniqNodeId && uniqNodeId.length > 0) {
                //in case we have a dw node id, then use query selector to get the element with attribute
                querySelectorString = '[' + DW_LIVEEDIT_CONSTANTS.DWUniqueId + '="' + uniqNodeId + '"]';
            } else if (selectorString && selectorString.length > 0) {
                querySelectorString = selectorString;
            }
            if (querySelectorString.length > 0) {
                try {
                    var elems = document.querySelectorAll(querySelectorString);
                    if (elems && elems.length > 0) {
                        targetElement = elems[0];
                    }
                } catch (e) {
                }
            }

            // When user clicks on a composite element and on adding contentEditability to that element,
            // browser sets cursor on a text node (child of the composite element) near to the cursor.
            // This selection goes to Dreamweaver and comes back to Live View. Hence ignore it but only if 
            // Live Edit is in progress. When Live Edit is NOT in progress then Keyboard Traversal can select
            // descendent of the current selected element but it is as expected.
            if(window.liveViewExtensionsObject.isLiveEditInProgress() && uniqNodeId !== window.liveViewExtensionsObject.getDwIdForSelectChildEvent()) {
                var element = targetElement;
                var curElement = window.liveViewExtensionsObject.getCurrentSelectedElement();
                while(element && element !== curElement) {
                    element = element.parentNode;
                }
                if(element)
                    return;
            }

            //if the selection is coming via code view selection,
            //then we need to remove marquee selection
            //And clicking on non contenteditable tags does not change the selection in browser,
            //so we need to clear the text selection in these cases
            if (window.getSelection() && (!document.hasFocus() || (targetElement && DW_LIVEEDIT_NON_CONTENT_EDITABLE_TAGS.includes(targetElement.tagName)))) {
                window.getSelection().removeAllRanges();
            }

            window.liveViewExtensionsObject.setCurrentSelectedElement(targetElement, elementIsNonEditable);
        };

        /*
        function:onDWThemeChange - This is the function DW calls (using evalJavascript)
                                     to notify all browsers that theme has changed

        Arguments: themeName - dark / light as of now
        Return: none
        */
        window.onDWThemeChange = function (themeName) {
            if (themeName && window.liveViewExtensionsObject) {
                window.liveViewExtensionsObject.setCurrentTheme(themeName);
                
                var dwEvent = {};
                dwEvent.type = DW_EXTENSION_EVENT.THEME_CHANGED;
                dwEvent.theme = themeName;
                window.liveViewExtensionsObject.messageToExtensions(dwEvent);
            }
        };
        
        /*
        function:DWEventToLiveView - This is the function DW calls (using evalJavascript)
                                     to send messages to LiveView

        Arguments: eventName - type of event (complete set of events @ Constants.js:DW_EVENTS)    
                   params - parameters for the event if any
        Return: none
        */
        window.DWEventToLiveView = function (eventName, params) {
            var dwEvent = {};
            dwEvent.type = eventName;
            if (params) {
                dwEvent.params = params;
            }
            //update live view selection if it is a selecion change
            if (eventName === DW_EVENTS.SelectionChanged) {
                if (params) {
                    var paramsArray = params.split(',');
                    if (paramsArray.length === 3) {
                        updateSelectedElement(paramsArray[0], paramsArray[1], paramsArray[2]);
                    }
                }
                window.liveViewExtensionsObject.setDwIdForSelectChildEvent(null);
            } else if (eventName === DW_EXTENSION_EVENT.LIVE_SURFACE_DRAG_START) {
                //update the dragged element if it is a live surface drag event
                window.liveViewExtensionsObject.setCurrentDraggedElement(params);
                //Make sure that liveDragFeedback extention is enabled
                window.liveViewExtensionsObject.enableLiveDragFeedback();
            }
			
			if (eventName === DW_EVENTS.ShowHelp) {
				var argObj = {};
                // check if any extension hud is visible
                var liveViewExtensionObject = window.liveViewExtensionsObject;
                var len = (liveViewExtensionObject) ? (liveViewExtensionObject.liveViewExtensionsArrayForHelp.length) : 0,
                    i = 0;
				for (i = 0; i < len; i++) {
					var extensionID = liveViewExtensionObject.liveViewExtensionsArrayForHelp[i].extensionId;
					if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.isExtensionVisible(extensionID)) {
						argObj.id = liveViewExtensionObject.liveViewExtensionsArrayForHelp[i].helpId;
						window.DWLECallJsBridgingFunction(DW_LIVEEDIT_CONSTANTS.Extensions, "DWSMLaunchHelp", argObj, false);
						break;
					}
				}
			}
			
			if (eventName === DW_EXTENSION_EVENT.HIDE_AUX_HUD || eventName === DW_EVENTS.PartialRefreshComplete) {
                if (typeof params !== "undefined") {
                    if (params === "true") {
                        dwEvent.commit = true;
                    }
                }
            }
			
			// Post this event to main window as well so that any modules
			// in main window can listen.
			// Example: We handle selection change message in SelectionSyncUtils.js
			window.postMessage(dwEvent, "*");

            //pass the event to extensions
            window.liveViewExtensionsObject.messageToExtensions(dwEvent);

            //should the extensions be enabled/disabled?
            if (eventName === DW_EVENTS.DisableExtensions) {
                window.liveViewExtensionsObject.disableAllExtensions(params);

                // Send "all extensions disabled" event to notify extensions
				// so that they can handle special cases.
                var disabledEvent = {};
                disabledEvent.type = DW_EXTENSION_EVENT.ALL_EXTENSIONS_DISABLED;
                window.postMessage(disabledEvent, "*");
            } else if (eventName === DW_EVENTS.EnableExtensions) {
                window.liveViewExtensionsObject.enableAllExtensions();
                
                //send a selection changed event to notify extensions
                var dwSelectionEvent = {};
                dwSelectionEvent.type = DW_EVENTS.SelectionChanged;
                window.liveViewExtensionsObject.messageToExtensions(dwSelectionEvent);
            }
            
            if (eventName === DW_EVENTS.PartialRefreshComplete || eventName === DW_EVENTS.StylesReloaded) {
                window.liveViewExtensionsObject.scheduleLoadingOfResponsiveFrameworkInfo();
            }
            
            if (eventName === DW_EVENTS.LiveResizeEnabled) {
                window.liveViewExtensionsObject.setLiveViewResizing(true);
            }
            if (eventName === DW_EVENTS.LiveResizeDisabled) {
                window.liveViewExtensionsObject.setLiveViewResizing(false);
            }
            
        };

        ResponsiveLayoutConstants.setWindowObj(window);
        window.liveViewExtensionsObject = new LiveViewObjectForExtensions();
        window.liveViewExtensionsObject.configureExtensionResourcePath();
        window.liveViewExtensionsObject.loadRequiredExtensions();
              
        /*
        function:shouldExtensionsHandleUndoRedo - Check whether UndoRedo commands should go to the Frame

        Arguments: none
        
        Return: true if the focus is in any of the editable areas in frame, false otherwise
        */
        window.shouldExtensionsHandleUndoRedo = function () {
            return window.isFocusInsideEditableAreaOfExtensions();
        };
        
        window.showHudForIAM = function () {
            return window.liveViewExtensionsObject.callShowHud();
        };

        window.IsTableExtensionVisible = function () {
            return window.liveViewExtensionsObject.isExtensionVisible("dwTableHud");
        };
		/*
        function:DWhasScroll - Check whether the scroll bar is visible
        
		Arguments : scroll direction ( vertical/ horizontal )
		
        Return: true if scroll bar is there
        */
		window.DWhasScroll = function (direction) {
            var element = document.body;
            direction = (direction === 'vertical') ? 'scrollTop' : 'scrollLeft';
            var result = !!element[direction];

            if (!result) {
                element[direction] = 1;
                result = !!element[direction];
                element[direction] = 0;
            }
            return result;
        };
		
        /*
        function:IsDragAllowed - Check whether the drag operation is should start from the click passed point

        Arguments: pointX, pointY
        
        Return: true if drag can be started , false otherwise
        */
        window.IsDragAllowed = function (pointX, pointY, offset) {
            var innerWidth = window.innerWidth;
            var innerHeight = window.innerHeight;
            var scrollHeight = document.body.scrollHeight;
			var scrollWidth = document.body.scrollWidth;
            var isHorScrollbarVisible = false;
            var isVerScrollbarVisible = false;
			
			isVerScrollbarVisible = window.DWhasScroll('vertical');
			isHorScrollbarVisible = window.DWhasScroll('horizontal');

			if ((isHorScrollbarVisible && (innerHeight - offset < pointY)) || (isVerScrollbarVisible && (innerWidth - offset < pointX))) {
                return false;
			}
             
            var canStartDrag = false;
            //give extensions a chance to cancel the drag
            //if 'canStartDragAtPoint' is defined for the extension, we will call that
            if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.liveViewExtensionsArray) {
                var i;
                var extensionsArray = window.liveViewExtensionsObject.liveViewExtensionsArray;
                for (i = 0; !canStartDrag && i < extensionsArray.length; i++) {
                    if (extensionsArray[i] && extensionsArray[i].enabled === true) {
                        var curFrame = extensionsArray[i].frame;
                        if (curFrame && curFrame.contentWindow && curFrame.contentWindow.canStartDragAtPoint) {
                            canStartDrag = curFrame.contentWindow.canStartDragAtPoint(pointX + window.scrollX, pointY + window.scrollY);
                        }
                    }
                }
            }

            return canStartDrag;
		};
        //Returns the src for an image tag.
        window.getImageSource = function () {
            if (window.liveViewExtensionsObject) {
                var curElem = window.liveViewExtensionsObject.getCurrentSelectedElement();
                if (curElem && curElem.nodeType === Node.ELEMENT_NODE) {
                    if (curElem.tagName === "IMG") {
                        return curElem.getAttribute("src");
                    }
                }
            }
            return "";
        };
        window.canSelectParent = function () {
            if (window.liveViewExtensionsObject) {
                var curElem = window.liveViewExtensionsObject.getCurrentSelectedElement();
                if (curElem && curElem.nodeType === Node.ELEMENT_NODE) {
                    if (curElem.tagName !== "BODY") {
                        return true;
                    }
                }
            }
            return false;
        };
        window.canSelectChild = function () {
            if (window.liveViewExtensionsObject) {
                var curElem = window.liveViewExtensionsObject.getCurrentSelectedElement();
                if (curElem && curElem.nodeType === Node.ELEMENT_NODE) {
                    if (curElem.firstElementChild) {
                        return true;
                    }
                }
            }
            return false;
        };
        
        window.setupResponsiveDefaultFunctions = function () {
            window.dwHiddenElementExist = false;
            window.hasHiddenElements = function () {
                if (window.liveEditFGInstance) {
                    return true;
                }

                return window.dwHiddenElementExist;
            };
            
            window.getAllInfoForResponsiveInsert = function () {
                var retObj = {};
                return retObj;
            };

            window.canElementBeHidden = function () {
                return false;
            };

            window.isElementHidden = function () {
                return false;
            };
            
        };
        window.setupResponsiveDefaultFunctions();
        
		/*
        function:isFocusInsideEditableAreaOfExtensions - Check whether if the focus is inside  editable region of iFrame
        Arguments: none        
        Return: true if the focus is in any of the editable areas in frame, false otherwise
        */
		window.isFocusInsideEditableAreaOfExtensions = function () {
			var activeElement = document.activeElement;
            if (!activeElement || activeElement.nodeType !== Node.ELEMENT_NODE) {
                return false;
            }
            if (activeElement.tagName === "IFRAME") {
                if (activeElement.id && window.liveViewExtensionsObject.isExtensionVisible(activeElement.id)) {
					if (activeElement.id === "dwSelectorHud") {
						return true;
					}
                    var activeElementInFrame = activeElement.contentDocument ? activeElement.contentDocument.activeElement : null;
                    if (activeElementInFrame && activeElementInFrame.nodeType === Node.ELEMENT_NODE) {
                        var activeElementTagName = activeElementInFrame.tagName;
                        // If this list grows keep it in constants.
                        if (activeElementTagName === "INPUT" || activeElementTagName === "SELECT" || activeElementTagName === "TEXTAREA") {
                            if (activeElementTagName === "INPUT" && (activeElementInFrame.classList.contains(DW_AUX_HUD_CONSTS.DwInputTypeNoUndo) === true)) { //for input type which has the class as InputTypeNoUndo,
                                //we let the live view handle undo and we update the HUD ui on selection change message
                                return false;
                            } else {
                                return true;
                            }
                        }
                        //check for attributes.
                        if (activeElementInFrame.getAttribute("contenteditable") === "true") {
                            return true;
                        }
                        var activeClass = activeElementInFrame.getAttribute('class');
                        var activeElementParentClass = activeElementInFrame.parentNode.getAttribute('class');
                        if (activeClass === DW_EDITABLE_SELECTORS_CONSTS.DwSelectorNameClassName && activeElementParentClass === DW_EDITABLE_SELECTORS_CONSTS.DwSelectorContainerClassName) {
                            return true;
                        }
                    }
                }
            }
            return false;
        };
		
		/*
        function:isAnyAuxHudHasFocus - Check whether if the focus is with the Image Hud
        Arguments: none        
        Return: true if the focus is in Image Hud, false otherwise
        */
        window.isAnyAuxHudHasFocus = function () {
            var activeElement = document.activeElement;
            if (activeElement && activeElement.tagName === "IFRAME") {
				if (activeElement.id && window.liveViewExtensionsObject.isExtensionVisible(activeElement.id)) {
					if ((activeElement.id === "dwImageHud") || (activeElement.id === "dwTextHud")) {
						return true;
					}
				}
			}
			return false;
		};
        
        
        /*
            function; getESHHudPlusButtonCoordinates - returns the coordinates of ESH Hud Plus Button
            Arguments: none
            return 
        */
        // this code can be part of editable Selector
        window.getESHHudPlusButtonCoordinates = function () {
            
            if (!window.liveViewExtensionsObject) {
                return;
            }
            var obj = {};
            obj.left = 0;
            obj.top = 0;
            obj.width = 0;
            obj.height = 0;
            obj.isVisible = false;
            var extensionIFrame = window.liveViewExtensionsObject.getExtensionIFrameById("editableSelectors");
            var selectedElement = window.liveViewExtensionsObject.getCurrentSelectedElement();
            
            if (extensionIFrame && selectedElement) {
                if (window.liveViewExtensionsObject.isElementEditable(selectedElement) && window.isESHVisible()) {
                    var wind = extensionIFrame.contentWindow;
                    var doc  = extensionIFrame.contentDocument;
                    if (doc) {
                        var addButton = doc.getElementById("addNewSelectorButton");
                        if (addButton) {
                            var boundingClientRect = addButton.getBoundingClientRect();
                            obj.left = boundingClientRect.left + wind.screenX - window.scrollX;
                            obj.top = boundingClientRect.top + wind.screenY - window.scrollY;
                            obj.width = boundingClientRect.right - boundingClientRect.left;
                            obj.height = boundingClientRect.bottom - boundingClientRect.top;
                            obj.isVisible = true;
                        }
                    }
                }
            }
            return obj;
        };

        /*
        function: isESHVisible - Check whether if ESH is visible
        Arguments: none
        Return: true if ESH is visible, false otherwise
        */
        window.isESHVisible = function () {
            return window.liveViewExtensionsObject.isExtensionVisible("editableSelectors");
        };
        
        
		/*
        function:isExtensionsHasFocus - Check whether if the focus is inside iFrame or focus is with the Image Hud
        Arguments: none        
        Return: true if the focus is in any of the editable areas in frame, false otherwise
        */
        window.isExtensionsHasFocus = function () {
            return window.isAnyAuxHudHasFocus() || window.isFocusInsideEditableAreaOfExtensions();
        };
		
        var messageHandlerForLiveEditExtensions = function (evt) {
            if (evt.data.dwExtensionEvent === true) {
                var property;
                for (property in DW_EXTENSION_EVENT) {
                    if (DW_EXTENSION_EVENT[property] === evt.data.type) {
                        window.liveViewExtensionsObject.messageToExtensions(evt.data);
                        break;
                    }
                }
                for (property in DW_PARENT_EVENT) {
                    if (DW_PARENT_EVENT[property] === evt.data.type) {
                        window.liveViewExtensionsObject.handleEventForParent(evt.data);
                        break;
                    }
                }
            }
        };
        
        window.addEventListener("message", messageHandlerForLiveEditExtensions, false);
		
		window.reInitializeLiveViewExtensions = function () {
			window.liveViewExtensionsObject.reInitializeLiveViewExtensions();
        };

		window.deInitializeLiveViewExtensions = function () {
			window.liveViewExtensionsObject.deInitializeLiveViewExtensions();
		};


        window.DW_Func_IsContextOnSelectorName = function () {
            var ret = "";
            var iframeElem = document.getElementById('editableSelectors');
            if (iframeElem && iframeElem.contentDocument) {
                var nodeList = iframeElem.contentDocument.querySelectorAll(':hover');
                if (nodeList) {
                    var findNode = function () {
                        //If there's a selector node in the hover hierarchy, return it.
                        var nlIndex = 0;
                        for (nlIndex = 0; nlIndex < nodeList.length; ++nlIndex) {
                            if (nodeList[nlIndex].classList.contains("selector") || nodeList[nlIndex].classList.contains("selectorNonEditable")) {
                                return nodeList[nlIndex];
                            }
                        }
                        return null;
                    };
                    var node = findNode();
                    if (node && node.nodeType === Node.ELEMENT_NODE && node.tagName.toLowerCase() === "dw-div") {
                        var elements = node.querySelectorAll('dw-div.selectorName');
                        if (elements && elements.length === 1) {
                            ret = elements[0].innerHTML;
                        }
                    }
                }
            }
            return ret;
        };

        window.DW_RefreshImageInLiveView = function (path) {
            var imgs = document.querySelectorAll('img[src*="' + path + '"]'),
                r = '?r=' + new Date().getTime(),
                i,
                src;
            
            if (0 !== imgs.length) {
                for (i = 0; i < imgs.length; i += 1) {
                    imgs[i].setAttribute('src', imgs[i].getAttribute('src') + r);
                }
                
                setTimeout(function () {
                    for (i = 0; i < imgs.length; i += 1) {
                        imgs[i].setAttribute('src', imgs[i].getAttribute('src').replace(r, ''));
                    }
                }, 4);
            }
        };
        
        window.DW_GetRectOfElementWithGivenID = function (id) {
            var results = document.querySelectorAll("[" + DW_LIVEEDIT_CONSTANTS.DWUniqueId + "='" + id + "']");
            var elem = results && results[0];
            if (elem) {
                window.ScrollElementIntoViewIfNeeded(id, "");
                var rect = elem.getBoundingClientRect();
                if (rect) {
                    var retString = rect.left + ',' + rect.top + ',' + rect.right + ',' + rect.bottom;
                    return retString;
                }
            }
            return '';
        };

        /*Given a node and offset within the node, return the offset from the start of the head tag in the document.*/
        
        var returnOffsetForGivenNode = function(node, offset) {
            var range = document.createRange(),
                offsetVal = 0,
                headElement = document.getElementsByTagName("head")[0],
                htmlElement = document.getElementsByTagName("html")[0];
            /* We are calculating offset relative to HEAD element because document.innerHTML doesnot provide content 
            like doctype, comments, whitespaces before "HEAD" element. So here we will calculate offsets relative to head.
            And correct offset of "HEAD" we will calculate in C++ side.*/
            if (range && headElement && htmlElement)
            {
                range.setStartBefore(headElement);
                range.setEnd(node, offset);
                var docFragment = range.cloneContents();
                var tmpNode = document.createElement("div");
                tmpNode.appendChild(docFragment);
                var selHTML = tmpNode.innerHTML;
                var count = 0, element = null, closingMarkupLength = 3; // </(*)>

                if (node.nodeType !== Node.ELEMENT_NODE)
                    element = node.parentNode;
                 else
                    element = node;

                /* InnerHTML provided by browser will be auto corrected for any selection. Appended by closing tags. So calculating closing tags length and                         trimming it from the total length.*/
                while (element && element !== htmlElement)
                {
                    if (element.tagName) 
                        count = count + element.tagName.length + closingMarkupLength;
                    element = element.parentNode;
                }
                offsetVal = selHTML.length-count;
            }
            return offsetVal;
        };
        
        /*
        function:DW_GetOffsetsForCurrentSelection - To get the offsets for current selection in browser.
        Return: start and end offsets in a string separated by comma
        */
        window.DW_GetOffsetsForCurrentSelection = function () {
            var offsets = "", startOffset = 0, endOffset = 0;
            var selectionObj = document.getSelection();
            
            if (selectionObj && selectionObj.anchorNode !== null && selectionObj.focusNode !== null)
            {
                var startOffset = returnOffsetForGivenNode(selectionObj.anchorNode, selectionObj.anchorOffset);
                var endOffset = returnOffsetForGivenNode(selectionObj.focusNode, selectionObj.focusOffset);

                if(startOffset < endOffset)
                    offsets = startOffset + "," + endOffset;
                else
                    offsets = endOffset + "," + startOffset;   // In case of reverse selection
            }
            return offsets;
        };
        
        window.DW_LiveEdit_LastLiveInsertPosition = "";
        window.DW_LiveEdit_SetInsertPosition = function (pos) {
            if (pos) {
                window.DW_LiveEdit_LastLiveInsertPosition = pos;
            }
        };
        
        window.DW_LiveEdit_GetInsertPosition = function () {
            if (window.DW_LiveEdit_LastLiveInsertPosition) {
                return window.DW_LiveEdit_LastLiveInsertPosition;
            }

            return "";
        };

        
    };

    initLiveViewExtensionsInfrastructure();
}());
//@ sourceURL=InitLiveViewExtensionsInfrastructure.js
