/*
 Copyright 2019 Adobe Systems Incorporated.  All rights reserved.
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/**
 * Utilitities
 */


/* Headlight Logging Data for In-Context Text Editing */
var DW_ICE_HEADLIGHTS = parent.DW_ICE_HEADLIGHTS;

function changeCurrentNodeContentToNBSP(currentNode) {
    'use strict';

    if (currentNode && !parent.DWTextEditUtility.elementContainTheTag(currentNode, "IMG") && parent.DWTextEditUtility.hasOnlyWhitespaceAsContent(currentNode)) {
        currentNode.innerHTML = parent.DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
        parent.DWTextEditUtility.setIbeamToGivenCharOffsetInAGivenNode(currentNode, 0);
    }
}

function figureOutPostProcessigAction (containerBL) {
    'use strict';

    var action = null;
    var tagIsInline = parent.DW_LIVE_TEXT_EDIT.InlineTextElements.includes(containerBL.tagName);
    var inListItemTag = (containerBL.tagName === "LI");

    if (parent.DWTextEditUtility.isCaretAtBeginningOfTextNode(containerBL)) {
        action = parent.ENTER_POST_PROCESSING_ACTIONS.ChangePreviousSiblingContentToNBSP;
    } else if (parent.DWTextEditUtility.isCaretAtEndOfTextNode(containerBL)) {
        if (inListItemTag) {
            action = parent.ENTER_POST_PROCESSING_ACTIONS.ChangeCurrentNodeContentToNBSP;
        } else {
            var needsPostProcessing = true;
            // if ibeam is currently at the end of in line tag, do post processing ONLY if it happens to be last child of its parent.
            if (tagIsInline === true) {
                if (containerBL.parentNode && containerBL.parentNode.lastChild &&
                    containerBL !== containerBL.parentNode.lastChild) {
                    needsPostProcessing = false;
                }
            }

            if (needsPostProcessing === true) {
                action = parent.ENTER_POST_PROCESSING_ACTIONS.ChangeCurrentNodeToParaAndChangeContentToNBSP;
            }
        }
    }
    return action;
}

// Performs post processing action identified by m_enterPostProcessingAction.
// This is called from onKeyUp i.e., after browsers enter key handling.
function performPostProcessingAction(action, nodeActual) {
	'use strict';

    if (action === null || !nodeActual) {
        return;
    }

    var innerHTML;
    if (action === parent.ENTER_POST_PROCESSING_ACTIONS.ChangeCurrentNodeContentToNBSP) {
        changeCurrentNodeContentToNBSP(nodeActual);
    } else if (action === parent.ENTER_POST_PROCESSING_ACTIONS.ChangePreviousSiblingContentToNBSP) {
        var previousSiblingNode = null,
            nodeActualsFirstBlockLevelParent = parent.DWTextEditUtility.getFirstBlockLevelParent(nodeActual);
        if (nodeActualsFirstBlockLevelParent) {
            previousSiblingNode = nodeActualsFirstBlockLevelParent.previousSibling;
        }

        if (previousSiblingNode && !parent.DWTextEditUtility.elementContainTheTag(previousSiblingNode, "IMG") && parent.DWTextEditUtility.hasOnlyWhitespaceAsContent(previousSiblingNode)) {
            previousSiblingNode.innerHTML = parent.DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
        }

        if (previousSiblingNode && previousSiblingNode.nextSibling && parent.DWTextEditUtility.elementContainTheTag(previousSiblingNode, "IMG")) {
            innerHTML = previousSiblingNode.nextSibling.innerHTML.trim();
            if (innerHTML === "" || innerHTML === "<br>") {
                previousSiblingNode.nextSibling.innerHTML = parent.DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
            }
        }
    } else if (action === parent.ENTER_POST_PROCESSING_ACTIONS.ChangeCurrentNodeToParaAndChangeContentToNBSP) {
        var targetElementToReplace = nodeActual;
        if (parent.DW_LIVE_TEXT_EDIT.InlineTextElements.indexOf(nodeActual.tagName) >= 0) {
            var firstBlockLevelParent = parent.DWTextEditUtility.getFirstBlockLevelParent(nodeActual);
            targetElementToReplace = firstBlockLevelParent ? firstBlockLevelParent : targetElementToReplace;

            // if targetElementToReplace is dw-span, then our actual node to replace is the div which got added browser.
            // start from nodeActual and go upwards till dw-span and break when you find a div which doesnot have data_liveedit_tagid
            if (targetElementToReplace && targetElementToReplace.tagName === parent.DW_LIVEEDIT_CONSTANTS.TextContainer) {
                var nodeIter = nodeActual;
                while (nodeActual !== targetElementToReplace) {
                    if (nodeIter.tagName === "DIV" && !nodeIter.hasAttribute(parent.DW_LIVEEDIT_CONSTANTS.DWUniqueId)) {
                        targetElementToReplace = nodeIter;
                        break;
                    }
                    nodeIter = nodeIter.parentNode;
                }
            }
        }

        if (targetElementToReplace) {
            if (!parent.DWTextEditUtility.elementContainTheTag(targetElementToReplace, "IMG") && parent.DWTextEditUtility.hasOnlyWhitespaceAsContent(targetElementToReplace)) {
                targetElementToReplace.innerHTML = parent.DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
            }

            // Unlike h1-h5, if enter is pressed at the end of H6 tag, browser doesn't create div, instead it creates H5. 
            if (targetElementToReplace.tagName === "DIV" || targetElementToReplace.tagName === "H6") {
                parent.document.execCommand('formatBlock', false, 'p');
            } else {
                parent.DWTextEditUtility.setIbeamToGivenCharOffsetInAGivenNode(targetElementToReplace, 0);
            }

            if (targetElementToReplace && targetElementToReplace.previousSibling && parent.DWTextEditUtility.elementContainTheTag(targetElementToReplace, "IMG")) {
                innerHTML = targetElementToReplace.previousSibling.innerHTML.trim();
                if (innerHTML === "" || innerHTML === "<br>") {
                    targetElementToReplace.previousSibling.innerHTML = parent.DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
                }
            }
        }
    }

    var selectionChangeEvent = new CustomEvent("selectionchange");
    parent.document.dispatchEvent(selectionChangeEvent);
}

function cleanUpContent(caret) {
    'use strict';

    if(!caret || !caret.isValid())
        return;

    var start = caret.getStartNode();
    var end = caret.getEndNode();
    parent.DWTextEditUtility.removeDuplicateIdAttributesInRange(start, end);
}

/** 
 * InsertBrHandler 
 * */
class InsertBrHandler extends BaseEventHandler {

    constructor() {
        'use strict';
        super();
    }

    onStart(event) {
        'use strict';

        event.preventDefault();
        event.stopPropagation();
    }

    onEnd(event) {
        'use strict';

        parent.DWTextEditUtility.insertHtmlAtCaretPosition('<br>');
        event.preventDefault();
        event.stopPropagation();
    }

    getAffectedElement() {
        'use strict';
        return super.getAffectedElement();
    }

    getEditRange(event) {
        'use strict';
        return this.getCurrentRange();
    }
}




/** 
 * EnterInEmptyBodyHandler 
 * */

class EnterInEmptyBodyHandler extends BaseEventHandler {

    constructor() {
        'use strict';
        super();
    }

    onStart(event) {
        'use strict';

        try {
            parent.document.execCommand("EnableUndoGrouping");

            this._createPTagsForEmptyBody();
            event.preventDefault();
            event.stopPropagation();

        } catch (e) {
        }
    }

    onEnd(event) {
        'use strict';
		parent.document.execCommand("DisableUndoGrouping");
    }

    getAffectedElement() {
        'use strict';
        return super.getAffectedElement();
    }

    getEditRange(event) {
        'use strict';
        return this.getCurrentRange();
    }

    // Creates two paragraph elements with 'nbsp' as content for empty body element.
    _createPTagsForEmptyBody() {
        'use strict';

        var firstNewNode = parent.document.createElement("p");
        var secondNewNode = parent.document.createElement("p");;
        firstNewNode.innerHTML = parent.DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
        secondNewNode.innerHTML = parent.DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
        firstNewNode = parent.document.body.insertBefore(firstNewNode, parent.document.body.firstElementChild);
        parent.document.body.insertBefore(secondNewNode, parent.document.body.firstElementChild);
        // Caret need to placed at the last child which is the firstNewNode.
        parent.DWTextEditUtility.setIbeamToGivenCharOffsetInAGivenNode(firstNewNode, 0);
    }
}



/** 
 * InsertNestedPHandler 
 * */
class InsertNestedPHandler extends BaseEventHandler {

    constructor(parent, start, end) {
        'use strict';

        super();
        this.m_parent = parent;
        this.m_start = start;
        this.m_end = end;
        this.m_editCaret = new Caret();
        this.m_editCaret.selectRange( this.getEditRange(), this.getAffectedElement());
    }

    onStart(event) {
        'use strict';

        try {
            parent.document.execCommand("EnableUndoGrouping");

            // Get current selection
            var selectionBnR = new SelectionBnR();
            selectionBnR.backup(parent.window.getSelection());

            // range.surroundContents needs start and end containers to be at same level
            // hence not using this.getEditRange().
            var range = parent.document.createRange();
            range.setStartBefore(this.m_start);
            range.setEndAfter(this.m_end);

            // Create new P element
            var element = parent.document.createElement('p');

            // Surround range with the newly created element
            range.surroundContents(element);

            // Restore selection so that default behaviour will work on it
            selectionBnR.restore(parent.window.getSelection());
        } catch (e) {
        }
    }

    onEnd(event) {
        'use strict';

        try {
            this._postProcess();
            cleanUpContent(this.m_editCaret);
        } finally {
            // Disable UNDO grouping.
            parent.document.execCommand("DisableUndoGrouping");
        }
    }

    getAffectedElement() {
        'use strict';
        return this.m_parent;
    }

    getEditRange(event) {
        'use strict';

        var range = parent.document.createRange();
        range.setStartBefore(this.m_start);
        if(this.m_end.nodeType === Node.TEXT_NODE)
            range.setEnd(this.m_end, this.m_end.nodeValue.length);
        else
            range.setEnd(this.m_end, Math.max(this.m_end.childNodes.length - 1, 0));
        return range;
    }

    _postProcess() {
        'use strict';

        if(!this.m_editCaret.isValid())
            return;

        var start = this.m_editCaret.getStartNode();
        var end = this.m_editCaret.getEndNode();

        for(var node = start; node; node = node.nextSibling) {

            // Make new nodes non-empty
            if (node.innerHTML === "<br>" || node.innerHTML.trim() === "") {
                node.innerHTML = parent.DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
            }

            if(node === end) {
                // Set IP to the beginning of the last node in the edit caret
                parent.DWTextEditUtility.setIbeamToGivenCharOffsetInAGivenNode(node, 0);
                break;
            }
        }

        var selectionChangeEvent = new CustomEvent("selectionchange");
        parent.document.dispatchEvent(selectionChangeEvent);
    }
}




/** 
 * InsertSiblingHandler 
 * */
class InsertSiblingHandler extends BaseEventHandler {

    constructor(parent, start, end) {
        'use strict';

        super();
        this.m_parent = parent;
        this.m_start = start;
        this.m_end = end;
        this.m_action = null;
        this.m_editCaret = new Caret();
    }

    onStart(event) {
        'use strict';

        try {

            // This handler changes current selected element in ESH. Changed element must
            // be selected before we determine editCaret. If changed element is BODY, then
            // Separator gets added. If editCaret is calculated before addition of the Separator
            // in BODY then resultant editCaret will have one more node than intended. 
            this.m_editCaret.selectRange( this.getEditRange(), this.getAffectedElement());

            parent.document.execCommand("EnableUndoGrouping");

            // Figure out post-processing action
            this.m_action = figureOutPostProcessigAction(this.m_parent);
        } catch (e) {
        }
    }

    onEnd(event) {
        'use strict';

        try {
            performPostProcessingAction(this.m_action, this.m_editCaret.getEndNode());
            cleanUpContent(this.m_editCaret);
        } finally {
            // Disable UNDO grouping.
            parent.document.execCommand("DisableUndoGrouping");
        }
    }

    getAffectedElement() {
        'use strict';
        return this.m_parent.parentNode;
    }

    getEditRange(event) {
        'use strict';

        var range = parent.document.createRange();
        range.setStartBefore(this.m_parent);
        range.setEnd(this.m_parent, Math.max(this.m_parent.childNodes.length - 1, 0));
        return range;
    }
}


/** 
 * InsertSiblingHandler 
 * */

class EnterHandler extends BaseEventHandler {

    constructor(shiftKey, smPath) {
        'use strict';

        super();
        this.m_shiftKey = shiftKey;
        this.m_smPath = smPath;
        this.m_handler = this._getHandler();
    }

    onStart(event) {
        'use strict';

        if(this.m_handler) {
            this.m_handler.onStart(event);
        } else if (event) {
            event.preventDefault();
            event.stopPropagation();
        }        
    }

    onEnd(event) {
        'use strict';

        if(this.m_handler) {
            this.m_handler.onEnd(event);
        } else if (event) {
            event.preventDefault();
            event.stopPropagation();
        }
    }

    getAffectedElement() {
        'use strict';

        if(this.m_handler)
            return this.m_handler.getAffectedElement();
        return null;
    }

    getEditRange(event) {
        'use strict';

        if(this.m_handler)
            return this.m_handler.getEditRange(event);
        return null;
    }

    _getHandler() {
        'use strict';

        var handler = null;
        var textSession = this._getTextSession();
        // 1. BR check
        if(this.m_shiftKey || (textSession && this._shouldInsertBrTag(textSession.parent))) {
            handler = new InsertBrHandler();
        }
        // 2. Empty body
        else if(this._isBodyEmpty()) {
            handler = new EnterInEmptyBodyHandler();
        }
        else {
            if(textSession && textSession.parent && textSession.start && textSession.end) {
                // 3. Insert nested P
                if (parent.DW_LIVE_TEXT_EDIT.TagsThatCreatesPTagsOnEnter.includes( textSession.parent.tagName)) {
                    handler = new InsertNestedPHandler(textSession.parent, textSession.start, textSession.end);
                }
                // 4. Insert siblings
                else {
                    var shouldInsertSibling = (textSession.parent.firstChild === textSession.start && textSession.parent.lastChild === textSession.end);
                    if (shouldInsertSibling ) {
                        if(this._isSMParentMatching(textSession.parent)) {
                            handler = new InsertSiblingHandler(textSession.parent, textSession.start, textSession.end);
                        }
                    } else {
                        handler = new InsertBrHandler();
                    }
                }
            }
        }
        if(handler && dwObject) {
            dwObject.logHeadlightsData(DW_ICE_HEADLIGHTS.ELV_LVE, DW_ICE_HEADLIGHTS.NewTagCreatedViaEnter);
            if(textSession && textSession.parent)
                dwObject.logHeadlightsData(DW_ICE_HEADLIGHTS.OTH_ELV, DW_ICE_HEADLIGHTS.lvePreString + textSession.parent.tagName + DW_ICE_HEADLIGHTS.lvePostStringTagSplit);
        }
        return handler;
    }

    _shouldInsertBrTag(node) {
        'use strict';

        if (!node || node.nodeType !== Node.ELEMENT_NODE) {
            return false;
        }

        if (parent.DW_LIVE_TEXT_EDIT.InsertBrTagOnEnter.includes(node.tagName)) {
            return true;
        }

        return false;
    }

    _isBodyEmpty() {
        'use strict';

        var range = this.getCurrentRange();
        if (range && range.commonAncestorContainer.tagName !== "BODY") {
            return false;
        }

        var childNodes = parent.document.body.childNodes;
        var length = childNodes.length;
        for (var i = 0; i < length; ++i) {
            var child = childNodes[i];
            if (child.nodeType === Node.TEXT_NODE && child.textContent.trim()){
                return false;
            } else if (child.nodeType === Node.ELEMENT_NODE) {
                if (child.getAttribute("id") === parent.DW_LIVEEDIT_CONSTANTS.UserDocSeparatorID || child.tagName.toLowerCase() === parent.DW_LIVEEDIT_CONSTANTS.GlobalContainerDiv) {
                    continue;
                } else {
                    return false;
                }
            }
        }
        return true;
    }

    _adjustCursorForTextNode(selection) {
        'use strict';

        if(!selection || !selection.isCollapsed) {
            return;
        }

        if(selection.anchorNode === selection.baseNode) {
            //If anchor Node and Base Node are same,
            //No Need to do anything.
            return;
        }

        //below is hack code which will make sure Cursor representation by Anchor Node and Base Node is same.
        if(selection.baseNode.nodeType === Node.TEXT_NODE && selection.baseNode.nodeValue !== "") {
            if(selection.anchorOffset === 0) {
                selection.modify("move", "forward", "character");
                selection.modify("move", "backward", "character");
            } else {
                selection.modify("move", "backward", "character");
                selection.modify("move", "forward", "character");
            }
        }
    }

    _getTextSession() {
        'use strict';

        var parentTS = null;
        var startTS = null;
        var endTS = null;

        // Allow ENTER key processing only if selection is collapsed and anchorNode is TEXT_NODE
        var selection = parent.document.getSelection();

        this._adjustCursorForTextNode(selection);
        var allow = selection && selection.isCollapsed && selection.anchorNode && selection.anchorNode.nodeType === Node.TEXT_NODE;
        if (allow) {

            var container = selection.anchorNode.childNodes[selection.anchorOffset]
                    || selection.anchorNode; //it's a text node

            var node = container;

            // Find block level parent
            while(node) {
                if(!this._isInline(node)) {
                    parentTS = node;
                    break;
                }
                node = node.parentNode;
            }

            // Move startTS and endTS to level just below parentTS
            node = container;
            while(node && node !== parentTS) {
                startTS = node;
                endTS = node;
                node = node.parentNode;
            }

            // Go to left-most text or inline-element
            node = startTS;
            while(node) {
                if(this._isInline(node)) {
                    startTS = node;
                } else {
                    break;
                }
                node = node.previousSibling;
            }

            // Go to right-most text or inline-element
            node = endTS
            while(node) {
                if(this._isInline(node) && 
                    (node.nodeType !== Node.ELEMENT_NODE || node.getAttribute("id") !== parent.DW_LIVEEDIT_CONSTANTS.UserDocSeparatorID)) {
                    endTS = node;
                } else {
                    break;
                }
                node = node.nextSibling;
            }
        }

        var textSession = {};
        textSession.parent = parentTS;
        textSession.start = startTS;
        textSession.end = endTS;
        return textSession;
    }

    // Check if 'node' and all its descendents are inline
    _isInline(node) {
        'use strict';

        // Non-elements are inline
        if(!node || node.nodeType !== Node.ELEMENT_NODE)
            return true;

        // Return early if its 'block' level
        if(!parent.DW_LIVE_TEXT_EDIT.InlineTextElements.includes(node.tagName))
            return false;

        // Check all descendents
        var childNodes = node.childNodes;
        var length = childNodes.length;
        for (var i = 0; i < length; ++i) {
            if(!this._isInline(childNodes[i]))
                return false;
        }

        // Its inline node
        return true;
    }

    _isSMParentMatching(element) {
        'use strict';

        var matching = true;
        do {

            if(!element || !element.parentNode || !this.m_smPath)
                break;
            
            // if parent tag is UL/OL, then there is chance of parent might have swaped in DW.
            // try commiting process with actualPrent.
            if (parent.DW_LIVE_TEXT_EDIT.TagsWhereRunsMaySwap.indexOf(element.parentNode.tagName) >= 0)
                break;

            var dwId = element.getAttribute( parent.DW_LIVEEDIT_CONSTANTS.DWUniqueId);
            if(!dwId)
                break;

            var index = this.m_smPath.indexOf(dwId);
            if(index < 0 || index >= this.m_smPath.length - 1)
                break;

            var parentIdSM = this.m_smPath[index + 1];
            var parentId = element.parentNode.getAttribute( parent.DW_LIVEEDIT_CONSTANTS.DWUniqueId);
            if(parentIdSM !== parentId)
                matching = false;

        } while(false);

        return matching;
    }
}